package usermutation

import (
	"log"
	"sync"
)

// WorkerManager is
type WorkerManager interface {
	InitializeWorker(shardID string, kinesisClient KinesisClient, checkpointStore CheckpointStore, callback ConsumerFunc)
	SpawnWorkers()
	Shutdown()
}

type workerManagerImpl struct {
	workers map[string]*worker
}

// NewWorkerManager allocates and returns a new WorkerManager
func NewWorkerManager() WorkerManager {
	return &workerManagerImpl{
		workers: make(map[string]*worker),
	}
}

func (w *workerManagerImpl) InitializeWorker(shardID string, k KinesisClient, c CheckpointStore, f ConsumerFunc) {
	worker := NewWorker(k, c, shardID, f)
	w.workers[shardID] = worker
}

func (m *workerManagerImpl) Shutdown() {
	var wg sync.WaitGroup
	for k, w := range m.workers {
		wg.Add(1)
		log.Printf("Shutting down worker %s", k)
		go func(w *worker) {
			w.Stop()
			wg.Done()
		}(w)
	}

	log.Printf("Waiting for workers...")
	wg.Wait()
}

func (m *workerManagerImpl) SpawnWorkers() {
	for k, w := range m.workers {
		go func(shardID string, worker *worker) {
			err := worker.Run()
			if err != nil {
				log.Printf("Worker %s died: %s", shardID, err.Error())
				// TODO actually handle this error, restart workers?
			} else {
				log.Printf("Worker %s terminated normally, not restarting", shardID)
			}
		}(k, w)
	}
}
