package backend

import (
	"fmt"
	"log"
	"strings"

	"golang.org/x/net/context"

	"code.justin.tv/web/user-mutation-stream-lib/usermutation"
)

type ChangeHandler interface {
	Handle(msg usermutation.Message) error
}

type changeHandlerImpl struct {
	classifier Classifier
	reporter   Reporter
	stats      Statter
	userReader UserReader
}

func NewChangeHandler(userReader UserReader, reporter Reporter, classifier Classifier, stats Statter) ChangeHandler {
	return &changeHandlerImpl{
		classifier: classifier,
		reporter:   reporter,
		stats:      stats,
		userReader: userReader,
	}
}

func (c *changeHandlerImpl) Handle(msg usermutation.Message) error {
	if msg.Event == "user_updated" {
		return c.handleChanged(msg)
	}
	return nil
}

func (c *changeHandlerImpl) handleChanged(msg usermutation.Message) error {
	if description, ok := msg.Changes["description"]; ok {
		if description.New != nil && description.New.(string) != "" {
			return c.handleBioChanged(msg)
		}
	}

	if status, ok := msg.Changes["status"]; ok {
		if status.New != nil && status.New.(string) != "" {
			return c.handleStatusChanged(msg)
		}
	}
	return nil
}

func (c *changeHandlerImpl) handleStatusChanged(msg usermutation.Message) error {
	c.incCounter("status.changed")

	ctx := context.Background()
	user, err := c.userReader.GetUserByID(ctx, msg.ID)
	if err != nil {
		log.Printf("could not retrieve login: %+v", err)
		return nil
	}
	if user.TOSViolation != nil && *user.TOSViolation {
		log.Printf("user banned, skipping")
		return nil
	}
	login := user.Login

	newStatus := msg.Changes["status"].New.(string)
	probBad := c.classifier.GetBadProbability(newStatus)
	log.Printf("status of user %s %.2f: %s", login, probBad, newStatus)

	matches := badStatusRegexp.FindAllStringSubmatch(newStatus, -1)
	if matches != nil {
		badWords := make([]string, 0, len(matches))
		for _, v := range matches {
			badWords = append(badWords, v[1])
		}
		content := fmt.Sprintf("suspected RuneScape phishing, bad words in title: %s | reporter: WIMPY TOS-o-Bot", strings.Join(badWords, ", "))

		c.incCounter("status.reported.words")
		//c.reporter.ReportChannelAsync(ctx, user.ID, ReportReasonStatus, content, nil)
		log.Print(content)
	}

	return nil
}

func (c *changeHandlerImpl) handleBioChanged(msg usermutation.Message) error {
	c.incCounter("bio.changed")

	ctx := context.Background()
	user, err := c.userReader.GetUserByID(ctx, msg.ID)
	if err != nil {
		log.Printf("could not retrieve login: %+v", err)
		return nil
	}
	if user.TOSViolation != nil && *user.TOSViolation {
		log.Printf("user banned, skipping")
		return nil
	}
	login := user.Login

	newBio := msg.Changes["description"].New.(string)
	bioClean := strings.Replace(newBio, "\n", " ", -1)
	bioClean = strings.Replace(bioClean, "\r", " ", -1)
	probBad := c.classifier.GetBadProbability(bioClean)
	log.Printf("bio of user %s %.2f: %s", login, probBad, bioClean)

	urlMatches := urlRegexp.FindAllStringSubmatch(newBio, -1)
	if urlMatches != nil {
		urls := make([]string, 0, len(urlMatches))
		for _, v := range urlMatches {
			urls = append(urls, v[1])
		}
		content := fmt.Sprintf("URLs in bio: %s | spam score: %.2f%% | reporter: WIMPY TOS-o-Bot", strings.Join(urls, ", "), probBad*100)
		if probBad >= minProbReport {
			c.incCounter("bio.reported.probability")
			//c.reporter.ReportChannelAsync(ctx, user.ID, ReportReasonBio, content, nil)
			log.Print(content)
			return nil
		}
	}

	matches := badBioRegexp.FindAllStringSubmatch(newBio, -1)
	if matches != nil {
		badWords := make([]string, 0, len(matches))
		for _, v := range matches {
			badWords = append(badWords, v[1])
		}
		content := fmt.Sprintf("bad words in bio: %s | reporter: WIMPY TOS-o-Bot", strings.Join(badWords, ", "))

		c.incCounter("bio.reported.words")
		c.reporter.ReportChannelAsync(ctx, user.ID, ReportReasonBio, content, nil)
	}

	return nil
}

func (c *changeHandlerImpl) incCounter(path string) {
	c.stats.IncCounter(path)
}
