package backend

import (
	"errors"
	"fmt"
	"log"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/web/user-mutation-stream-lib/usermutation"
	"golang.org/x/net/context"
)

const panelMonitorCheckpointName = "panel-monitor"

type PanelMonitor interface {
	Monitor() error
}

type panelMonitorImpl struct {
	checkpointStore usermutation.CheckpointStore
	classifier      Classifier
	panelReader     PanelReader
	reporter        Reporter
	stats           Statter
	userReader      UserReader
}

func NewPanelMonitor(userReader UserReader, panelReader PanelReader, reporter Reporter, classifier Classifier, stats Statter, checkpointStore usermutation.CheckpointStore) PanelMonitor {
	return &panelMonitorImpl{
		checkpointStore: checkpointStore,
		classifier:      classifier,
		panelReader:     panelReader,
		reporter:        reporter,
		stats:           stats,
		userReader:      userReader,
	}
}

func (p *panelMonitorImpl) Monitor() error {
	checkpoint, err := p.checkpointStore.GetCheckpoint(panelMonitorCheckpointName)
	if err != nil {
		return err
	}
	if checkpoint == nil {
		return errors.New("Panel checkpoint not found")
	}

	startID, err := strconv.Atoi(*checkpoint)
	if err != nil {
		return err
	}

	for {
		nextID, err := p.scanPanelsFrom(startID)
		if err != nil {
			log.Printf("error while scanning panels: %+v", err)
		}

		if nextID == startID {
			time.Sleep(1 * time.Minute)
		} else {
			startID = nextID
			time.Sleep(5 * time.Second)
		}
	}

}

func (p *panelMonitorImpl) scanPanelsFrom(startID int) (int, error) {
	ids := make([]string, 0, 100)
	for id := startID; id < startID+100; id++ {
		ids = append(ids, strconv.Itoa(id))
	}

	ctx := context.Background()
	panelResponse, err := p.panelReader.GetPanelsByIDs(ctx, ids, nil)
	if err != nil {
		log.Printf("could not retrieve panels: %+v", err)
		return startID, err
	}

	if panelResponse == nil {
		return startID, nil
	}

	nextStartID := 0
	for _, panel := range panelResponse.Panels {
		p.incCounter("panel.scanned")

		nextStartID = panel.ID + 1
		panelText := fmt.Sprintf("%s: %s", panel.Data.Title, panel.Data.Description)
		panelTextClean := strings.Replace(panelText, "\n", " ", -1)
		panelTextClean = strings.Replace(panelTextClean, "\r", " ", -1)
		probBad := p.classifier.GetBadProbability(panelTextClean)
		log.Printf("panel %d of user %d %.2f: %s", panel.ID, panel.UserID, probBad, panelTextClean)
		containsLinks := panel.Data.Link != "" || urlRegexp.MatchString(panelTextClean)

		if containsLinks && probBad >= minProbReport {
			user, err := p.userReader.GetUserByID(ctx, panel.UserID)
			if err != nil {
				log.Printf("could not retrieve login: %+v", err)
				continue
			}
			if user.TOSViolation != nil && *user.TOSViolation {
				log.Printf("user banned, skipping")
				continue
			}

			content := fmt.Sprintf("bad panel | spam score: %.2f%% | reporter: WIMPY TOS-o-Bot", probBad*100)
			p.incCounter("panel.reported.probability")
			//p.reporter.ReportChannelAsync(ctx, user.ID, ReportReasonPanel, content, nil)
			log.Print(content)
		}
	}

	err = p.checkpointStore.SaveCheckpoint(panelMonitorCheckpointName, strconv.Itoa(nextStartID))
	if err != nil {
		log.Printf("could not persist checkpoint: %+v", err)
	}

	return nextStartID, nil
}

func (p *panelMonitorImpl) incCounter(path string) {
	p.stats.IncCounter(path)
}
