package backend

import (
	"encoding/json"
	"fmt"
	"net/http"
	"strings"

	"golang.org/x/net/context"

	"code.justin.tv/common/twitchhttp"
)

type PanelResponse struct {
	Panels []Panel `json:"panels"`
}

type Panel struct {
	ID     int       `json:"id"`
	UserID int       `json:"user_id"`
	Kind   string    `json:"kind"`
	Data   PanelData `json:"data"`
}

type PanelData struct {
	Link        string `json:"link"`
	Image       string `json:"image"`
	Title       string `json:"title"`
	Description string `json:"description"`
}

type PanelReader interface {
	GetPanelsByUserID(ctx context.Context, userID int, reqOpts *twitchhttp.ReqOpts) (*PanelResponse, error)
	GetPanelsByIDs(ctx context.Context, ids []string, reqOpts *twitchhttp.ReqOpts) (*PanelResponse, error)
}

type panelReaderImpl struct {
	twitchhttp.Client
}

func NewPanelReader(conf twitchhttp.ClientConf) (PanelReader, error) {
	twitchClient, err := twitchhttp.NewClient(conf)
	if err != nil {
		return nil, err
	}

	return &panelReaderImpl{twitchClient}, nil
}

func (r *panelReaderImpl) GetPanelsByUserID(ctx context.Context, userID int, reqOpts *twitchhttp.ReqOpts) (*PanelResponse, error) {
	path := fmt.Sprintf("/v1/users/%d/panels", userID)
	return r.getPanels(ctx, path, reqOpts)
}

func (r *panelReaderImpl) GetPanelsByIDs(ctx context.Context, ids []string, reqOpts *twitchhttp.ReqOpts) (*PanelResponse, error) {
	path := fmt.Sprintf("/v1/panels?ids=%s", strings.Join(ids, ","))
	return r.getPanels(ctx, path, reqOpts)
}

func (r *panelReaderImpl) getPanels(ctx context.Context, path string, reqOpts *twitchhttp.ReqOpts) (*PanelResponse, error) {
	req, err := r.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{})
	httpResp, err := r.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}
	defer func() {
		err = httpResp.Body.Close()
	}()

	switch httpResp.StatusCode {
	case http.StatusOK:
		var decoded PanelResponse
		if err := json.NewDecoder(httpResp.Body).Decode(&decoded); err != nil {
			return nil, err
		}
		return &decoded, nil
	case http.StatusBadRequest:
		// No panels found
		return nil, nil
	default:
		// Unexpected result
		return nil, fmt.Errorf("Unexpected status code %d", httpResp.StatusCode)
	}
}
