package backend

import (
	"bytes"
	"encoding/json"
	"fmt"
	"log"
	"net/http"
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/common/twitchhttp"
)

type ReportReason string

const (
	ReportReasonBio    ReportReason = "spam"
	ReportReasonPanel  ReportReason = "spam"
	ReportReasonStatus ReportReason = "spam"
	minProbReport                   = 0.99
	reportToken                     = "JciEQf0Z-8G764yQVhhImgxjPsLStxvA"
	reportFromUserID                = "140176432"
)

type Reporter interface {
	ReportChannelAsync(ctx context.Context, userID int, reason ReportReason, content string, reqOpts *twitchhttp.ReqOpts)
	ReportChannel(ctx context.Context, userID int, reason ReportReason, content string, reqOpts *twitchhttp.ReqOpts) error
}

type reporterImpl struct {
	twitchhttp.Client
	stats Statter
}

func NewReporter(conf twitchhttp.ClientConf, stats Statter) (Reporter, error) {
	twitchClient, err := twitchhttp.NewClient(conf)
	if err != nil {
		return nil, err
	}

	return &reporterImpl{twitchClient, stats}, nil
}

func (r *reporterImpl) ReportChannel(ctx context.Context, userID int, reason ReportReason, content string, reqOpts *twitchhttp.ReqOpts) error {
	reportParams := map[string]interface{}{
		"authorisation_token": reportToken,
		"report": map[string]interface{}{
			"from_user_id":   reportFromUserID,
			"target_user_id": fmt.Sprintf("%d", userID),
			"content":        "user_report",
			"reason":         reason,
			"description":    content,
			"origin":         "wimpy/tos-o-bot",
		},
	}

	body, err := json.Marshal(reportParams)
	if err != nil {
		return err
	}

	path := fmt.Sprintf("/reports")
	req, err := r.NewRequest("POST", path, bytes.NewBuffer(body))
	if err != nil {
		return err
	}

	req.Header.Add("Content-Type", "application/json")

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{})
	httpResp, err := r.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer func() {
		err = httpResp.Body.Close()
	}()

	switch httpResp.StatusCode {
	case http.StatusOK:
		return nil

	default:
		// Unexpected result
		return fmt.Errorf("Unexpected status code %d", httpResp.StatusCode)
	}

}

func (r *reporterImpl) ReportChannelAsync(ctx context.Context, userID int, reason ReportReason, content string, reqOpts *twitchhttp.ReqOpts) {
	go func() {
		err := retry(3, func() error {
			log.Printf("sending report - %d - %s", userID, content)
			r.incCounter("service.reporter.attempts")
			return r.ReportChannel(ctx, userID, ReportReasonBio, content, reqOpts)
		})
		if err != nil {
			log.Printf("could not report user: %+v", err)
			r.incCounter("service.reporter.errors")
		}
	}()
}

func (r *reporterImpl) incCounter(path string) {
	r.stats.IncCounter(path)
}

func retry(attempts int, callback func() error) (err error) {
	for i := 0; i < attempts; i++ {
		err = callback()
		if err == nil {
			return nil
		}

		time.Sleep(20 * time.Second)

		log.Println("retrying...")
	}
	return fmt.Errorf("after %d attempts, last error: %s", attempts, err)
}
