package main

import (
	"log"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/kinesis"

	"code.justin.tv/chat/golibs/clients/usersservice"
	"code.justin.tv/common/config"
	"code.justin.tv/common/gometrics"
	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/ncaspar/tos-o-bot/backend"
	"code.justin.tv/web/user-mutation-stream-lib/usermutation"
)

const (
	reporterMaxIdleConnsPerHost     = 10
	usersServiceMaxIdleConnsPerHost = 10
	panelsMaxIdleConnsPerHost       = 10
)

var (
	classifierPersistRate = 10 * time.Second
)

func main() {
	config.Register(map[string]string{
		"aws-region":         "us-west-2",
		"dynamo-table-name":  "nils-tos-o-bot",
		"reporter-host":      "https://leviathan.internal.twitch.tv",
		"panels-host":        "http://panels.production.us-west2.twitch.tv/",
		"users-service-host": "http://users-service.prod.us-west2.twitch.tv",
		"bayes-db-path":      "/opt/twitch/tos-o-bot/shared/bayesian.db",
	})

	err := config.Parse()
	if err != nil {
		log.Fatal(err)
	}

	workerManager := usermutation.NewWorkerManager()
	kinesisClient := kinesis.New(session.New(), aws.NewConfig().WithRegion(config.Resolve("aws-region")))
	dynamoDBClient := dynamodb.New(session.New(), aws.NewConfig().WithRegion(config.Resolve("aws-region")))
	checkpointStore := usermutation.NewDynamoDBCheckpointStore(dynamoDBClient, config.Resolve("dynamo-table-name"), config.Environment())

	stats := configureStatter()

	usersServiceClient, err := configureUsersServiceClient()
	if err != nil {
		log.Fatal(err)
	}

	userReader := configureUserReader(usersServiceClient)

	panelReader, err := configurePanelReader()
	if err != nil {
		log.Fatal(err)
	}

	reporter, err := configureReporter(stats)
	if err != nil {
		log.Fatal(err)
	}

	classifier, err := configureClassifier()
	if err != nil {
		log.Fatal(err)
	}

	panelMonitor := configurePanelMonitor(userReader, panelReader, reporter, classifier, stats, checkpointStore)
	go func() {
		err := panelMonitor.Monitor()
		log.Printf("error while monitoring panels: %+v", err)
	}()

	gometrics.Monitor(config.Statsd(), time.Second*5)

	handler := configureChangeHandler(userReader, reporter, classifier, stats)

	consumer := usermutation.NewStreamConsumer(kinesisClient, checkpointStore, workerManager, handler.Handle)
	log.Fatal(consumer.Start())
}

func configurePanelMonitor(userReader backend.UserReader, panelReader backend.PanelReader, reporter backend.Reporter, classifier backend.Classifier, stats backend.Statter, checkpointStore usermutation.CheckpointStore) backend.PanelMonitor {
	return backend.NewPanelMonitor(userReader, panelReader, reporter, classifier, stats, checkpointStore)
}

func configureStatter() backend.Statter {
	return backend.NewStatter(config.Statsd())
}

func configureUsersServiceClient() (usersservice.Client, error) {
	transport := twitchhttp.TransportConf{
		MaxIdleConnsPerHost: usersServiceMaxIdleConnsPerHost,
	}
	conf := twitchhttp.ClientConf{
		Host:      config.Resolve("users-service-host"),
		Transport: transport,
	}
	return usersservice.NewClient(conf)
}

func configureUserReader(usersServiceClient usersservice.Client) backend.UserReader {
	return backend.NewUserReader(usersServiceClient)
}

func configureReporter(stats backend.Statter) (backend.Reporter, error) {
	transport := twitchhttp.TransportConf{
		MaxIdleConnsPerHost: reporterMaxIdleConnsPerHost,
	}
	conf := twitchhttp.ClientConf{
		Host:      config.Resolve("reporter-host"),
		Transport: transport,
	}
	return backend.NewReporter(conf, stats)
}

func configurePanelReader() (backend.PanelReader, error) {
	transport := twitchhttp.TransportConf{
		MaxIdleConnsPerHost: panelsMaxIdleConnsPerHost,
	}
	conf := twitchhttp.ClientConf{
		Host:      config.Resolve("panels-host"),
		Transport: transport,
	}
	return backend.NewPanelReader(conf)
}

func configureChangeHandler(userReader backend.UserReader, reporter backend.Reporter, classifier backend.Classifier, stats backend.Statter) backend.ChangeHandler {
	return backend.NewChangeHandler(userReader, reporter, classifier, stats)
}

func configureClassifier() (backend.Classifier, error) {
	return backend.NewClassifier(config.Resolve("bayes-db-path"))
}
