# TwitchFeatureIngestionClient

A Python client to load features into Online Feature Store

# Installation

See documentation on how to use the [internal pypi repository](https://git.xarth.tv/dta/docs/blob/master/artifactory/using-artifactory.md#pypi-repository) hosted in artifactory.
Steps: 
- update the pip.conf index-url to include the internal repository
- including the package in requirements.txt as any other pip package

# Prerequisites

The feature ingestion client depends on the feature registry to first obtain metadata about the feature before beginning to ingest feature values into OFS

- Make sure the account using this client has been added to the allowlist for [Feature Registry](https://code.amazon.com/packages/TwitchVXFeatureRegistryCDK/blobs/mainline/--/lib/stacks/accounts.ts)
  - If it doesn't exist, please ping on the #vx-feature-store channel and someone will add the account for you
- Make sure the role running the service and using this client has the following policy attached to it. Replace the resource with `"arn:aws:s3:::mlfs-feature-metadata-beta/*` for your non-prod environments
```
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Effect": "Allow",
            "Action": "s3:GetObject",
            "Resource": "arn:aws:s3:::mlfs-feature-metadata-prod/*"
        }
    ]
}
```


# Usage

```bash
pip install TwitchFeatureIngestionClient
```

- Put features API (updates a single feature value for an entity id, doesn't overwrite or update feature values if it's not present in the request)

```python
from octarine.clients.TwitchFeatureIngestionClient import *
from botocore.client import Config as BotoConfig

config = Config(AWS_REGION="us-west-2", ENVIRONMENT="beta")
feature_configs = [FeatureKey("vxs", "test-feature-1", "1"), FeatureKey("vxs", "test-feature-2", "2")]

# Optional, the client relies on a default ddb config if not provided
ddb_config = BotoConfig(retries={'max_attempts': 10,'mode': 'standard'})
fic = TwitchFeatureIngestionClient(config, feature_configs, ddb_client_config=ddb_config)

entity_set = [FeatureEntity(ENTITY_CHANNEL, "123")]
feature_vals = {feature_configs[0]: [3.1231, 6.54], feature_configs[1]: "feature-string-val"}
fic.put_features(entity_set, feature_vals)

# With optional params
fic.put_features(entity_set, feature_vals, ttl=300, disable_metrics=True)
```

- Batch write features API (updates all the feature values for an entity id, if the feature is not present in the request it's value in DDB is overwritten to a null value)
```python
from octarine.clients.TwitchFeatureIngestionClient import *
from botocore.client import Config as BotoConfig

config = Config(AWS_REGION="us-west-2", ENVIRONMENT="beta")
feature_configs = [FeatureKey("vxs", "test-feature-1", "1"), FeatureKey("vxs", "test-feature-2", "2")]

# Optional, the client relies on a default ddb config if not provided
ddb_config = BotoConfig(retries={'max_attempts': 10,'mode': 'standard'})
fic = TwitchFeatureIngestionClient(config, feature_configs, ddb_client_config=ddb_config)

feature_records = [
    FeatureRecord(
        [FeatureEntity(ENTITY_CHANNEL, "123")],
        {feature_configs[0]: [3.1231, 6.54], feature_configs[1]: "feature-string-val"},
    ),
    FeatureRecord(
        [FeatureEntity(ENTITY_CHANNEL, "456")],
        {feature_configs[0]: [0.834], feature_configs[1]: "random-string-value"},
    ),
]
fic.batch_write_features(feature_records)

# With optional params
fic.batch_write_features(feature_records, ttl=300)
```

# Integration test

This client's integration tests between FIC and OFS run as part of the pipeline: https://pipelines.amazon.com/pipelines/TwitchFICVerification

This client's integration tests between FIC and FSC run as part of the pipeline: https://pipelines.amazon.com/pipelines/TwitchMLFSVerification

If you are making changes that might have risks of incompatibilities with other MLFS component, consider to add some test cases. If the change relate to how data is stored in OFS, please add test cases [here](https://code.amazon.com/packages/TwitchFICVerificationTests/trees/mainline/--/). If the change could affect how data is read from OFS or relate to feature schema, please add test cases [here](https://code.amazon.com/packages/TwitchMLFSVerification/blobs/mainline/--/cmd/TwitchMLFSVerification/fic_end_2_end_integration_test.go)


# To release a new version of this client

1. Verify that [TwitchFICVerification pipeline](https://pipelines.amazon.com/pipelines/TwitchFICVerification) and[TwitchMLFSVerification pipeline](https://pipelines.amazon.com/pipelines/TwitchMLFSVerification) are running successully with the latest changes.
2. Go into `submodule` directory, and run `git pull` to get the latest Schema proto files
3. Run `make generate` to generate new python code form the schema file.
4. Merge the approved branch to master and use `make tag` to draft a new release
5. Build the Tag in Jenkins:
   - Navigate to the tag builds of [feature ingestion client](https://jenkins-og.xarth.tv/job/octarine-twitchfeatureingestionclient/view/tags/)
   - Click on the icon to build the new tag under the `Build On` section
   - The build will automatically push the new version to Artifactory. The version name will match the tag.
