from enum import Enum
from typing import Dict, Optional, Union

import boto3


class MetricUnit(str, Enum):
    Seconds = "Seconds"
    Microseconds = "Microseconds"
    Milliseconds = "Milliseconds"
    Percent = "Percent"
    Count = "Count"
    CountPerSecond = "Count/Second"


class MetricsSender(object):
    """
    MetricsSender abstracts sending metrics to cloudwatch
    """

    def __init__(
        self,
        region_name: str,
        namespace: str,
        base_dimensions: Optional[Dict[str, str]] = None,
    ):
        self.client = boto3.client("cloudwatch", region_name=region_name)
        self.namespace = namespace
        self.base_dimensions = base_dimensions

    def send(
        self,
        metric_name: str,
        value: Union[int, float],
        unit: MetricUnit,
        dimensions: Optional[Dict[str, str]] = None,
    ):
        """
        Send a custom metric to CloudWatch.

        Usage:

        ```
        metrics = MetricsSender('feature_ingestion_beta', base_dimensions={'Stage': settings.ENVIRONMENT})
        metrics.send('metrics_name', 0.563, unit=MetricUnit.Percent)
        ```
        """
        if dimensions is None:
            dimensions = {}

        dimensions_list = []
        for k, v in ({**self.base_dimensions, **dimensions}).items():
            dimensions_list.append({"Name": k, "Value": v})

        if len(dimensions_list) > 10:
            dimensions_list = dimensions_list[:10]

        self.client.put_metric_data(
            Namespace=self.namespace,
            MetricData=[
                {
                    "MetricName": metric_name,
                    "Dimensions": dimensions_list,
                    "Values": [value],
                    "Unit": unit.value,
                }
            ],
        )
