import functools

from boto3 import Session
from botocore.credentials import RefreshableCredentials
from botocore.session import get_session as botocore_get_session


def get_credentials(role_arn: str, sts_client):
    role = sts_client.assume_role(RoleArn=role_arn, RoleSessionName="ofs-write-role")
    credentials = role["Credentials"]
    if credentials is None:
        raise PermissionError(
            "Role arn: " + role_arn + " could not assumed to write to OFS"
        )

    return {
        "access_key": credentials.get("AccessKeyId"),
        "secret_key": credentials.get("SecretAccessKey"),
        "token": credentials.get("SessionToken"),
        "expiry_time": credentials.get("Expiration").isoformat(),
    }


def get_session(role_arn: str, aws_region: str, sts_client):
    refresh_credentials = functools.partial(get_credentials, role_arn, sts_client)
    session_credentials = RefreshableCredentials.create_from_metadata(
        metadata=refresh_credentials(),
        refresh_using=refresh_credentials,
        method="sts-assume-role",
    )

    session = botocore_get_session()
    session._credentials = session_credentials
    session.set_config_variable("region", aws_region)
    refresh_session = Session(botocore_session=session)
    return refresh_session
