import logging


def escape_string(input_str):
    return input_str.replace("\\", r"\\").replace('"', r"\"")


class LoggerFormatter(logging.Formatter):
    format_str = 'time=%(asctime)s level=%(levelname)s name="%(name)s" module="%(module)s" msg="%(message)s"'
    datefmt_str = "%Y-%m-%dT%H:%M:%SZ"

    def __init__(self):
        logging.Formatter.__init__(self, self.format_str, self.datefmt_str)

    def format(self, record):
        record.name = escape_string(record.name)
        record.module = escape_string(record.module)
        if isinstance(record.msg, str):
            record.msg = escape_string(record.msg)
        elif hasattr(record.msg, "__str__"):
            record.msg = escape_string(record.msg.__str__())
        return logging.Formatter.format(self, record)

    def add_format_fields(self, fields):
        for field in fields:
            self.format_str += ' {field}="%({field})s"'.format(field=field)
        logging.Formatter.__init__(self, self.format_str, self.datefmt_str)


def _get_module_logger(severity, name, extra=None):
    """Get a logger object."""
    logger = logging.getLogger(name)

    # no need to propagate logs. Prevents from duplicate log lines.
    logger.propagate = False

    # stderr outputting handler
    handler = logging.StreamHandler()
    formatter = LoggerFormatter()
    handler.setFormatter(formatter)
    logger.addHandler(handler)
    logger.setLevel(severity)

    # add custom fields
    logger = logging.LoggerAdapter(logger, extra)
    return logger
