---
title: Delete & Access Requests
weight: 3
---
Make sure you've gone through [PDMS onboarding]({{< ref "pdms_onboarding" >}}) before starting these workflows.

## Deletion requests
Services that are required to respond to deletion requests, as indicated in [Catalog](https://catalog.xarth.tv/services), must onboard their data to receive data deletion requests from PDMS. For existing services, responding to deletion requests will default to required in Catalog.

When a deletion request begins, [EventBus](https://git.xarth.tv/pages/eventbus/docs/) will broadcast on the `UserDestroy` channel. Service owners must subscribe to this event and respond to events accordingly. Additionally, PDMS will record the deletion request for each individual service required to respond, for auditing purposes.

To onboard your data to receive deletion requests:

1. Integrate a PDMS Service Client:
    ```go
    package pdms

    import (
    "code.justin.tv/amzn/PDMSLambdaTwirp"
    twirplambdatransport "code.justin.tv/amzn/TwirpGoLangAWSTransports/lambda"
    "github.com/aws/aws-sdk-go/aws"
    "github.com/aws/aws-sdk-go/aws/credentials"
    "github.com/aws/aws-sdk-go/aws/endpoints"
    "github.com/aws/aws-sdk-go/aws/session"
    "github.com/aws/aws-sdk-go/service/lambda"
    )

    func New() PDMSLambdaTwirp.PDMSService {
    sess := session.Must(session.NewSessionWithOptions(session.Options{
        Config: aws.Config{
            Region:              aws.String("us-west-2"),
            STSRegionalEndpoint: endpoints.RegionalSTSEndpoint,
            // We want a long timeout for PDMS
            // Lambda based services can take a while to warm up on cold start
            HTTPClient: &http.Client{Timeout: 10 * time.Second},
        },
    }))
    creds := stscreds.NewCredentials(sess, "<pdms-caller-role>")
    pdmsLambdaCli := lambda.New(sess, &aws.Config{Credentials: creds})
    pdmsTransport := twirplambdatransport.NewClient(pdmsLambdaCli, "<pdms-lambda-arn>")
    pdms := PDMSLambdaTwirp.NewPDMSServiceProtobufClient("", pdmsTransport)

    return pdms
    }
    ```
    Your implementation details may vary. See [Passport](https://git.xarth.tv/identity/passport/blob/master/clients/pdms/pdms.go), and [GDS](https://git.xarth.tv/gds/gds/blob/f7547449bfa3447ca5d130011cda505cb2636f9f/vhs/cmd/rewinder/pdms_s2s_client.go) for example implementations.
1. Subscribe to the [UserDestroy](https://git.xarth.tv/eventbus/schema/blob/master/pkg/user/destroy.pb.go) EventBus event. For information on how to use eventbus see [eventbus subscriber docs](https://git.xarth.tv/pages/eventbus/docs/subscribers/)
1. Listen for `UserDestroy` Events:
    ```go
    func ListenEventbus() error {
        // create eventbus mux as dispatcher
        mux := eventbus.NewMux()

        // register handler for UserDestroy event
        user.RegisterDestroyHandler(mux, processUserDestroyEvent)

        awsSession := session.New(&aws.Config{
            Region:              "<region>",
            STSRegionalEndpoint: "<endpoint>",
        })

        _, err := sqsclient.New(sqsclient.Config{
            Session:    awsSession,
            Dispatcher: mux.Dispatcher(),
            QueueURL:   "<queue url>",
        })
        if err != nil {
            return err
        }

        return nil
    }
    ```

Your service will now be notified when users submit deletion requests.

### Handling deletion requests
When your service receives `UserDestroy` events, process the event, then notify PDMS that you've deleted the user data or have promised to delete the data in the future:
```go
func processUserDestroyEvent(ctx context.Context, h *eventbus.Header, event *user.UserDestroy) error {
    // Grab the user ID from the eventbus message
    userID := event.UserId

    // 
    // Delete User Data
    //

    // Notify PDMS all data for this user ID has been deleted
    err := pdmsClient.ReportDeletion(ctx, &PDMSLambdaTwirp.ReportDeletionRequest{
        UserId:    userID,
        ServiceId: "<service id>",
    })
    if err != nil {
        return err
    }
    
    // Notify PDMS that all data for this user ID will be deleted by timestamp
    err := pdmsClient.ReportDeletion(ctx, &PDMSLambdaTwirp.ReportDeletionRequest{
        UserId:    userID,
        ServiceId: "<service id>",
        Timestamp: "<future timestamp>",
    })
    if err != nil {
        return err
    }

    return nil
}
```

For an example, [see how this was implemented for Passport](https://git.xarth.tv/identity/passport/pull/1416/files#diff-1e02c2c8f1b95788f54acda33ce2f83b).

## Access requests
{{% notice note %}} Onboarding for access requests is still under development. 
{{% /notice %}}

PDMS Access allows you to onboard your stored user data, in order for it to be returned to the user.
Since a single service can have many sub-services/datastores/tables, PDMS thinks in terms of individual pieces of data and does not rely on a one-to-one mapping of data and services. This is unlike deletion requests where PDMS has a 1-1 mapping of service ID to deletion request.
 
PDMS also allows for multiple files per `data_ID` to be uploaded. This gives your service maximum flexibility on how data should be organized/uploaded.

To onboard your data to receive access requests:

1. Audit your service's datastores to determine what data needs to be returned in a DSAR access request by referring to the [Data Subject Access Requests (DSAR) guidelines](https://docs.google.com/spreadsheets/d/1r9nNb9WKggaSoI-SUDNKQHbuYnAXrx22A99u0hzlqZE/edit?ouid=103465881923364140819&usp=sheets_home&ths=true). For services with lots of entries per user, PDMS allows you to define a single `data_ID` for that entry and upload
multiple entries for a single request. For example, for Whispers it makes sense to upload a single file per conversation
which aligns with how the data is stored on the backend as well as providing a more intuitive UX.
1. Create proto files for each individual section of data in [`pdms-schema`](https://git.xarth.tv/privacy/pdms-schema). When creating your proto files, follow these guidelines:
    * No enums in the proto, since we will ultimately be converting these to a json format when delivering the data to the user. Enums will render as their integer representation.
    * Avoid long lists of objects. If your service has to return many objects, define a single object in the proto and then use the Multifile Upload API.
    * No internal identifiers, (for example, TUID is external).
    * An example protobuf file:
        ```protobuf
        syntax = "proto3";
        package example.v1;
        import "google/protobuf/timestamp.proto";
        option go_package = "datasets/example/examplev1";

        // data-id: example-data-id
        // Representation of a person.
        message Person {
        // Unique ID number for this person.
        string id = 1;
        // Name of the person.
        string name = 2;
        // Email address associated to the person.
        string email = 3;

        // Phone number and type.
        message PhoneNumber {
            string number = 1;
            string type = 2;
        }

        // Phones associated with a person.
        repeated PhoneNumber phones = 4;

        google.protobuf.Timestamp last_updated_time = 5;
        }

        // List of persons.
        message AddressBook {
        repeated Person people = 1;
        }

        ```
1. Register your proto files by adding them into the map PDMS uses to file outstanding requests. This also involves providing meta information for the `data_ID` that will be rendered in the report to the end user:
    ```go
    func GetInfo() *MetaInfo {
        m := make(map[string]MetaInformation)
        // data id
        m["example-data-id"] = MetaInformation{
            // Service ID that owns the data
            ServiceID: "-1",
            // a title for the piece of data
            Title: "title",
            // a long description for the data to inform the data on individual fields and how they are used
            Description: "description",
            // the proto message type so we know what to deserialize it into
            Message: reflect.TypeOf(examplev1.Person{}),
            // Where the files get placed in the final report
            ReportPath: "test/example",
        }
        return &MetaInfo{m}
    }
    ```
1. Have Privacy review proto files by sharing the schema branch with the Privacy team. This is to ensure you are returning all the data required and to prevent Twitch from leaking data. Contact @natalieYu or @hoang for help with reviewing your proto files.
1. Onboard to the user DSAR request EventBus event. PDMS will fire off an [EventBus event](https://eventbus.xarth.tv/events/UserDataRequestCreate) for each DSAR request. This event contains the `user_ID` as well as a the `request_ID` for this individual request. Your workflow may vary depending on the nature of the access request:
    * **No user data**: If your service does not have any data on the user, call `ReportServiceAccessUpload` with `hasData = false` or
    `CompleteMultiFileAccess` with an expected count of `0`. Otherwise, collect the data for the user or call `GetOutstandingServiceUploadRequest` which will return all outstanding requests for a given `data_ID`.
    * **SingleFile Upload**: Call `ReportServiceAccessUpload` with `hasData = true`. PDMS will return a presented s3 url to upload the proto to. The filename
    of the file will be its `data_ID`. Once you upload the data, PDMS will automatically mark the `data_ID` as done.
    * **Multifile Upload**: Call `GenerateMultiFileUrls` with a list of the filenames for each file. Each filename must be unique. 
    PDMS will then return a list of signed urls. This API can be called multiple times. 
    Once all the data has been uploaded to s3, call `CompleteMultiFileAccess` with the expected number of files uploaded. 
    PDMS will validate that the expected number matches the count files in s3, then mark the `data_ID` as complete. 
    PDMS does not care how many files it generates for a `data_ID`, only how many files are in the s3 bucket at the time of completion.
1. Call PDMS to check if your data is required for an access requset using the `DataRequired` API. If your data is not required you can drop the message. 
1. Upload your data to PDMS with a `POST` to the provided signed url for each file. Each upload link is valid for 15 min from time of creation:
    ```go
    package pdms

    import (
        "fmt"
        "io"
        "io/ioutil"
        "net/http"
        "time"
    )

    func uploadFileToS3(s3Url string, payload io.Reader) error {
        // upload the file to s3
        httpReq, err := http.NewRequest("PUT", s3Url, payload)
        if err != nil {
            return err
        }
        client := &http.Client{
            Timeout: time.Second * 10,
        }

        httpResponse, err := client.Do(httpReq)
        if err != nil {
            return err
        }

        bodyBytes, err := ioutil.ReadAll(httpResponse.Body)
        if err != nil {
            return err
        }

        if httpResponse.StatusCode != http.StatusOK {
            return fmt.Errorf("non 200 status code from s3 msg=%s code=%d", string(bodyBytes), httpResponse.StatusCode)
        }
    }

    ```
1. Submit a PR to `pdms-schema` with the data added to it.
