# == Class: cgroups
#
# For precise nodes or newer, installs and configures cgroup support
# Default package is cgroup-lite
#
# === Parameters
#
# [*basedir*]
#  Base mount point of cgroup.  Default: /sys/fs/cgroup
#
# [*namespace*]
#  Namespace for services to live inside cgroup basedir.  Default: system
#
# [*package*]
#  Package name to support cgroups.  Default: cgroup-lite
#   * Module was written to assume cgroup-lite.  There may be slight mods to use other packages like
#      libcgroup + cgroup-bin
#
# [*service*]
#  Name of service provided from package.  Default: cgroup-lite
#
class cgroups (
  $basedir   = $cgroups::params::cgroups_basedir,
  $namespace = $cgroups::params::namespace,
  $package   = $cgroups::params::package,
  $service   = $cgroups::params::service,
) inherits cgroups::params {
  validate_absolute_path($basedir)
  validate_string($namespace)

  $allcgroups        = split($::available_cgroups,',')
  $cgroups_path      = prefix($allcgroups,"${basedir}/")
  $cgroups_namespace = suffix($cgroups_path,"/${namespace}")

  # Cgroup for precise or newer
  if ($::lsbmajdistrelease >= 12) {
    if !defined(Cgroups::Namespace[$namespace]) {
      cgroups::namespace{ "DEFAULT: ${namespace}":
        paths   => $cgroups_namespace,
        require => [Package[$package],Service[$service],],
      }
    }

    package{ $package:
      ensure => 'present',
    }

    service{ $service:
      ensure  => 'running',
      enable  => true,
      require => Package[$package],
    }
  } else {
    notify{ "Unsupported: cgroups not supported on ${::lsbdistcodename}":
      loglevel => info,
    }
  }
}
