# == Definition: cgroups::service
#
#  Creates the directories in the $basedir needed to enable cgroups for a service for
#   supported versions.  Must be precise or newer.  If an older version of ubuntu runs this
#   a message will be sent that cgroups is not supported
#
# == Parameters
#
# [*basedir*]
#  Base mount point of cgroup.  Default: /sys/fs/cgroup
#
# [*namespace*]
#  Namespace for services to live inside cgroup basedir.  Default: system
#
# [*rules*]
#  Hash of rules that will be enforced for the app.  Default: {}
#   ex. {
#         'memory' => {
#           'memory.swappiness' => 1,
#                     },
#         'blkio'  => {
#           'blkio.weight' => 1000,
#         }
#        }
#
# [*cgroups*]
#  Array of cgroups to enable for this service. Default: ['cpuacct','memory']
#   - Array is checked against all available cgroups and will log error if
#     cgroup is not enabled and not continue configuring any cgroups
#
define cgroups::service (
  $basedir   = '/sys/fs/cgroup',
  $namespace = 'service',
  $rules     = {},
  $cgroups   = [
    'cpuacct',
    'memory',
  ],
) {
  validate_absolute_path($basedir)
  validate_array($cgroups)
  validate_hash($rules)
  validate_string($namespace)

  if ($::lsbmajdistrelease >= 12) {
    require 'cgroups'

    # Facter < 2.0 does not support types.  The fact was written to return a ,
    # delimited string from an array
    $allcgroups = split($::available_cgroups,',')

    # Validate that the cgroups that we want enabled are available on this machine
    if member($allcgroups,$cgroups) {
      $cgroups_path      = prefix($cgroups,"${basedir}/")
      $cgroups_namespace = suffix($cgroups_path,"/${namespace}")
      $cgroups_app       = suffix($cgroups_namespace,"/${name}")

      # Writes a shell script that will execute the cgroup rules for a service
      # This is to survive reboots and puppet not running before supervise starts the job
      file{ "/usr/local/bin/cgroups_rules_${name}.sh":
        ensure  => present,
        owner   => 'root',
        group   => 'root',
        mode    => '0755',
        content => template('cgroups/cgroups_rules.sh.erb'),
        notify  => Exec["run cgroups rules for ${name}"],
      }

      # We want to be able to make live changes and not have to bounce the service
      exec{ "run cgroups rules for ${name}":
        command     => "/usr/local/bin/cgroups_rules_${name}.sh",
        user        => 'root',
        refreshonly => true,
        require     => [
          File["/usr/local/bin/cgroups_rules_${name}.sh"],
          File[$cgroups_app],
        ],
      }

      cgroups::namespace{ "${name} ${namespace}":
        paths => $cgroups_namespace,
      }

      # Creates directories in $basedir/<cgroup>/$name
      # We want to be able to make live changes and not have to bounce the service
      file{ $cgroups_app:
        ensure  => directory,
        owner   => 'root',
        group   => 'root',
        mode    => '0755',
        require => Cgroups::Namespace["${name} ${namespace}"],
      }
    } else {
      notify{ "${name} cgroup unavailable":
        loglevel => err,
        message  => "cgroup unavailable, available cgroups: ${::available_cgroups}",
      }
    }
  } else {

    notify{ "${name} No cgroups for you...":
      loglevel => info,
      message  => "Cgroup support for precise or newer.  You are running ${::lsbdistcodename}",
    }
  }
}
