# == Define: courier::install
#
# Installs Twitch software using courier
#
# === Parameters:
# [*env*]
#  Environment used as part of consul query to get current release for that environment, Default: *required*
#
# [*repo*]
#  Github repo project name, Default: *required*
#
# [*basedir*]
#  Twitch base directory, Default: <hiera:twitch_basedir>
#
# [*type*]
#  Type of install, Options: tar, docker, Default: tar
#
# [*image*]
#  Required when type => docker, name of image, Default: undef
#
# [*user*]
#  User to run courier and own directories, Default: jtv
#
# [*group*]
#  Group to own directories, Default: jtv
#
# [*consul_host*]
#  Consul host to connect to, Default: localhost:8500
#
# [*extra_flags*]
#  Extra flags to pass to courier command.  Must be a string and begin with --, Default: ''
#  Example: '--option1 value1 --option2 value2'
#
# [*enable_courierd*]
#  Enable courierd monitoring of shasums for service, Default: false
#
# === Example
# This example will install owl in the default locations with the version listed for production
# Courier will extract owl artifact to /opt/twitch/owl/releases/<sha> and move the symlink
#  to point to this revision
#
# courier::install{ 'owl':
#   env  => 'production',
#   repo => 'web/owl',
# }
#
# Example courier command:
#  courier tar install --consul-host localhost:8500 --repo 'web/owl' --environment production --dir /opt/twitch/owl
define courier::install (
  $env,
  $repo,
  $basedir         = hiera('twitch_basedir','/opt/twitch'),
  $type            = 'tar',
  $image           = undef,
  $user            = 'jtv',
  $group           = 'jtv',
  $consul_host     = hiera('consul_host', 'localhost:8500'),
  $extra_flags     = '',
  $enable_courierd = false,
) {
  include courier
  include courierd::config
  include consul

  $appdir = "${basedir}/${name}"

  if ($type == 'docker') and ($image == undef) {
    fail('FAIL: courier::install type => docker requires image to be set')
  }

  if ($extra_flags != '') and ($extra_flags !~ /^--/) {
    fail("FAIL: courier::install extra_flags does not begin with --")
  }

  if ($type != 'docker') and ($type != 'tar') {
    fail('Not a valid type.  Valid types: docker, tar')
  }

  courierd::path{ $name:
    path   => $appdir,
    enable => $enable_courierd,
  }

  $type_flags = $type ? {
    'docker' => "--image ${image}",
    'tar'    => "--dir ${appdir}",
    default  => '',
  }

  $courier_command = "courier ${type} install --consul-host ${consul_host} --repo ${repo} --environment ${env} ${type_flags} ${extra_flags}"

  courier::dirs{ $name:
    basedir => $basedir,
    user    => $user,
    group   => $group,
  }

  exec{ "courier install ${name}":
    command => $courier_command,
    path    => ['/bin', '/sbin', '/usr/bin', '/usr/sbin', '/usr/local/bin', '/usr/local/sbin'],
    user    => $user,
    timeout => 600,
    require => [
      Class['courier'],
      Courier::Dirs[$name],
    ],
  }
}
