# == Define: supervise
#
# Create a service that is monitored by daemontools.
#
# === Parameters
#
# [*wd*]
#   Working directory to run the command in.
#
# [*daemon*]
#   Binary to run. This is run via exec so it should be be a single command
#   containing no pipes or parameters.
#
# [*options*]
#   Options and parameters to pass to the daemon.
#
# [*syslog*]
#   Configures daemontools to log to syslog.
#
#   Valid options:
#     local   = Logs to /var/log/jtv.log
#     local2  = Alias for local
#     local3  = Logs to /var/log/jtv/<tagname>.log
#     remote  = Logs to central syslog
#
# [*predaemon*]
#   Shell commands to run prior to the daemon. Usually for exporting shell parameters.
#
# [*user*]
#   Before 'exec'ing to the daemon the user will be changed to the defined value.
#
# [*down*]
#   Valid values:
#   true  => ensure a down file exists for the service
#   force => ensure a down file exists for the service and make sure the service is down in supervise
#   false => remove the down file if it exists
#   undef => don't change anything
#
# [*tagname*]
#   Set the tagname to log to syslog under. By default this is set to $name.
#
# [*fdlimit*]
#   Change the file descriptor limit set by ulimit. By default this is 262144.
#
# [*daemon_dir*]
#   The directory where daemon files exist (contrary to where daemontools looks for them)
#
# [*physical_cd*]
#   Forces cd to move to the actual location of any symlinks. By default set to false.
#
# [*enable_cgroups*]
#   Bool that will call the cgroups define to create cgroup for this service,
#     default to false
#   Valid values:
#    true  => calls cgroups define to create cgroup directories for this service
#    false => will not call cgroup define but will not remove if they exist
#
# [*cgroups_basedir*]
#   Base directory of cgroups.  Not likely to change
#    Default: /sys/fs/cgroup
# [*cgroups_rules*]
#  Accepts a hash of hashes
#   Default: {}
#   Ex: { 
#         'memory' => {
#           'memory.swappiness'     => '0',
#           'memory.limit_in_bytes' => '9223372036854775807',
#          },
#          'blkio' => {
#            'blkio.weight' => '1000',
#          },
#        }
#
# [*cgroups*]
#   Array of cgroups to enable if enable_cgroups = true
#     Default: ['cpuacct','memory']
#
# [*cgroups_namespace*]
#   Sets the namespace under which the service will live in the cgroups
#     Default: service
#     Results in /sys/fs/cgroup/<cgroup>/$cgroups_namespace/$name
#    
# === Examples
#
#   daemontools::supervise { 'echo_server':
#     daemon  => "/usr/bin/nc",
#     options => "-l 5000",
#     user    => "nobody",
#     wd      => "/tmp/"
#   }
#
define daemontools::supervise (
  $wd,
  $daemon,
  $physical_cd       = false,
  $options           = '',
  $syslog            = undef,
  $predaemon         = [],
  $user              = '',
  $down              = '',
  $tagname           = undef,
  $fdlimit           = 262144,
  $daemon_dir        = '/etc/service',
  $enable_cgroups    = false,
  $cgroups_basedir   = '/sys/fs/cgroup',
  $cgroups_rules     = {},
  $cgroups_namespace = 'service',
  $cgroups           = [
    'cpuacct',
    'memory',
  ],
) {
  require daemontools
  validate_bool($physical_cd)

  validate_bool($enable_cgroups)
  validate_array($predaemon,$cgroups)
  validate_absolute_path($daemon_dir,$cgroups_basedir)
  validate_hash($cgroups_rules)

  $add_logging = $syslog ? {
    local   => "local2",
    local2  => "local2",
    local3  => "local3",
    remote  => "local1",
    default => ""
  }

  if $tagname {
    $tag = $tagname
  } else {
    $tag = $name
  }

  $runas = $user ? {
    /^\w+/ => "setuidgid ${user} ",
    default => '',
  }

  # lets make sure that we're using a dir that is on the whitelist
  if !($daemon_dir in $daemontools::params::daemon_dirs) {
    $join_dirs = join($daemontools::params::daemon_dirs, ', ')
    fail("${daemon_dir} is not one of '${join_dirs}'")
  }

  $service_dir  = "${daemon_dir}/${name}"
  $service_run  = "${service_dir}/run"
  $service_down = "${service_dir}/down"

  $cgroups_rules_cmd    = "/usr/local/bin/cgroups_rules_${name}.sh"
  $cgroups_paths        = prefix($cgroups,"${cgroups_basedir}/")
  $cgroups_paths_svc    = suffix($cgroups_paths,"/${cgroups_namespace}/${name}")
  $cgroups_procs_svc    = suffix($cgroups_paths_svc,'/cgroup.procs')
  $cgroups_add          = prefix($cgroups_procs_svc,'/bin/echo $$ > ')
  $cgroups_dirs_cmd     = prefix($cgroups_paths_svc,'mkdir -p ')

  if ($enable_cgroups) and ($::lsbmajdistrelease >= 12) {
    $real_enable_cgroups = true
  } else {
    $real_enable_cgroups = false
  }

  if ($real_enable_cgroups) {
    # Make sure all $cgroups are available before we go further
    $allcgroups = split($::available_cgroups,',')
    $valid_cgroups = member($allcgroups,$cgroups)

    if ($valid_cgroups) {
      cgroups::service{ $name:
        cgroups   => $cgroups,
        basedir   => $cgroups_basedir,
        rules     => $cgroups_rules,
        namespace => $cgroups_namespace,
      }
    } else {
      $string_cgroups = join($cgroups,',')
      notify{ "${name}: cgroup unavailable":
        loglevel => err,
        message  => "cgroup unavailable for Daemontools::Supervise[${name}], available cgroups: ${::available_cgroups}, provided cgroups: ${string_cgroups}",
      }
    }
  }

  # Injecting cgroup related steps before other $predaemon settings
  # if cgroups are enabled and all elements in $cgroups are available
  # If neither of these are true, we remove cgroup related lines from the run file
  if ($real_enable_cgroups) and ($valid_cgroups) {
    $predaemon_full = flatten([$cgroups_dirs_cmd,$cgroups_rules_cmd,$cgroups_add,$predaemon])
  } else {
    $predaemon_full = $predaemon
  }

  file { $service_dir:
    ensure  => 'directory',
    require => File[$daemon_dir]
  }

  file { $service_run:
    ensure  => 'present',
    mode    => 0755,
    content => template("${module_name}/supervise_run.erb"),
    require => File[$service_dir],
  }

  # this is here for a migration path for
  # older services in /etc/service dir
  # this will create a reloadable/notifiable service
  if ($daemon_dir == '/var/lib/service') {
    service { $name:
      subscribe => File[$service_run],
      provider  => 'daemontools',
      path      => $daemon_dir
    }
  }

  # create log dirs/files if we need them
  if $add_logging {
    $log_dir = "${service_dir}/log"
    $log_run = "${log_dir}/run"

    $before_svc = $daemon_dir ? {
      '/var/lib/service' => Service[$name],
      default            => undef,
    }

    file { $log_dir:
      ensure  => 'directory',
      require => File[$service_dir],
      before  => $before_svc,
    }

    file { $log_run:
      ensure  => 'present',
      mode    => 0755,
      content => "#!/bin/bash\n\nexec logger -p ${add_logging}.info -t ${tag}\n",
      require => File[$log_dir],
      before  => $before_svc,
    }
  }

  case $down {
    /^(true|force)$/: {
      file { $service_down:
        content  => '',
        owner    => 'root',
        group    => 'root',
        mode     => 0644,
        require  => File[$service_dir],
        before   => File[$service_run],
      }

      # if force down and we dont have a service
      if ($down == "force") and !defined(Service[$name]) {
        exec { "/usr/bin/svc -dk $service_dir":
          require => File[$service_run]
        }
      }
      $service_ensure = 'stopped'
    }
    false: {
      file { $service_down:
        ensure  => 'absent',
        require => File[$service_dir]
      }
      $service_ensure = 'running'
    }
    # since this is not actually boolean
    # we have to have a default case for the service
    default: {
      $service_ensure = 'running'
    }
  }

  # if we have the service defined from above
  # we can override the state it should be in
  if defined(Service[$name]) {
    Service[$name] { ensure => $service_ensure }
  }
}
