#!/usr/bin/env python
import argparse
import requests
import hmac, hashlib, base64
import urllib, urlparse
import time
import operator
from xml.dom import minidom
from sys import exit, stderr

class Services(object):
    def __init__(self, key, secret):
        self.__get_services__(key, secret)


    def __get_services__(self, key, secret):
        base_path = "/services/cdn/v1.0/"
        access_group = "96847"

        path = base_path + "%s/" % access_group
        body = ''

        r = make_request(path, body, key, secret)
        self.xml = minidom.parseString(r.text)
        self.services = {}

        for service in self.xml.getElementsByTagName("service"):
            service_id = service.getAttribute("id")
            for ni in service.getElementsByTagName("ni"):
                name = ni.getAttribute("id")
                
                self.services[name] = {"active": True, "scid": service_id}


    def get_scid(self, name):
        if not self.services.has_key(name):
            return None

        return self.services[name]["scid"]


    def list(self):
        sorted_services = sorted(self.services.iteritems(), key=operator.itemgetter(0))
        format = "%-50s %-10s %-6s"

        print format % ("Service Name", "SCID", "Active")
        for service, info in sorted_services:
            print format % (service, info["scid"], info["active"])

    def __str__(self):
        return self.xml.toprettyxml()


def main():
    global services, args
    parser = argparse.ArgumentParser()
    parser.add_argument("-v", "--verbose", action="store_true", help="verbose output")
    parser.add_argument("-t", "--test", action="store_true", help="test api key")
    parser.add_argument("-d", "--domains", action="append", default=[], help="domains to purge from")
    parser.add_argument("-k", "--key", default="255203756", help="api key")
    parser.add_argument("-s", "--secret", default="XWRWrTyJ56RBt3DMHsN2", help="api secret")
    parser.add_argument("-l", "--list", action="store_true", help="list active services")
    parser.add_argument("uris", nargs='*', help="uris to purge, must start with '/'")
    args = parser.parse_args()

    services = Services(args.key, args.secret)

    if args.test:
        test_key(args.key, args.secret)
    elif args.list:
        services.list()
        exit(0)
    else:
        if len(args.uris) == 0:
            print >>stderr, "ERROR: Require uris to purge"
            exit(-1)

        if len(args.domains) == 0:
            print >>stderr, "ERROR: Requrie domains to purge"
            exit(-1)

        for uri in args.uris:
            if not uri.startswith('/'):
                print >>stderr, "Invalid uri '%s'" % uri
                exit(-1)

        purge_items(args.uris, args.domains, args.key, args.secret, verbose=args.verbose)

def make_request(path, body, key, secret, method="get", host="https://ws.level3.com"):
    """
    See Level3's API docs[1] for more information on how to construct the security headers.

    Specifically the "Request signature form" section.
    
    1: https://mediaportal.level3.com/cdnServices/help/Content/API/API_KeysSecurity.htm
    """
    global args
    content_type = "text"
    content_md5 = hashlib.md5(body).hexdigest()

    headers = {
        'Date' : time.strftime('%a, %d %b %Y %H:%M:%S +0000', time.gmtime()),
        'Content-Type': content_type,
        'Content-MD5': content_md5,
    }

    canon = '\n'.join([
        headers['Date'],
        path,
        headers['Content-Type'],
        method.upper(),
        headers['Content-MD5'],
    ])

    h = hmac.new(secret, canon, hashlib.sha1)
    sig = base64.encodestring(h.digest()).strip()
    headers['Authorization'] = "MPA %s:%s" % (key, sig)

    if args.verbose:
	print "URL: %s%s" % (host, path)

    fn = getattr(requests, method.lower())
    r = fn(host + path, data=body, headers=headers)

    if not r.ok:
        print "Failed: '%s' with status '%s'" % (r.reason, r.status_code)
        if r.text:
            print r.text
        exit(-1)

    return r

def purge_items(items, domains, key, secret, ignore_case=False, force=False, verbose=False):
    global services
    """
    See Level3's API docs[1] for more information about the 'invalidation' endpoint.

    1: https://mediaportal.level3.com/cdnServices/help/Content/API/API_Specs/Inval_POST.htm
    """
    base_path = "/invalidations/v1.0"
    access_group = "96847"

    for domain in domains:
        scid = services.get_scid(domain)
        path = "%s/%s/%s/%s" % (base_path, access_group, scid, domain)

        for item in items:
            body = """<paths>\n  <path>%s</path>\n</paths>""" % item

            r = make_request(path, body, key, secret, method="POST")
            xml = minidom.parseString(r.text)
            invalidation = xml.getElementsByTagName("invalidations")[0].firstChild
            invalidation_id = invalidation.getAttribute("id")
            print "%s %s" % (domain + item, invalidation_id)

def test_key(key, secret):
    """
    See Level3's doc[1] for more infromation about the 'key' endpoint.

    1: https://mediaportal.level3.com/cdnServices/help/Content/API/API_Specs/Key.htm
    """
    path="/key/v1.0"

    r = make_request(path, '', key, secret)
    xml = minidom.parseString(r.text)
    print xml.toprettyxml()

if __name__ == "__main__":
    main()
