#!/usr/bin/env python

"""
Use killstrays as a wrapper for programs that may create stray child
processes, especially when running them under supervise.

Programs written with python multiprocessing often fall into this
category.

If killstrays catches one of the HANDLED_SIGS, it passes that signal
along to all (and only) its descendent processes.

Usage
-----

If before you said:

foobar.py foo bar baz

now you can say:

killstrays.py foobar.py foo bar baz
"""


import signal
import sys
import os
import subprocess
import atexit
import re
import logging


HANDLED_SIGS = (signal.SIGTERM, signal.SIGHUP, signal.SIGINT)


logging.basicConfig(level=logging.INFO)
mypid = os.getpid()
logging.info('pid %s', mypid)

try:
    # create a new pgrp
    os.setpgrp()
except OSError:
    # unless it's already a group leader, in which case ignore...
    logging.info('setpgrp() failed (already pgrp leader?)')

sigrev = dict([(v, k) for k, v in signal.__dict__.items() if re.match(r'SIG[A-Z]', k)])

def sighandler(signum=None, frame=None):
    """Kill children and exit, either because of a signal or atexit..."""
    logging.debug('caught %s' % (sigrev[signum],))
    os.killpg(mypid, signum)
    sys.exit(0)

def exithandler():
    os.killpg(mypid, signal.SIGTERM)
    

for sig in HANDLED_SIGS:
    signal.signal(sig, sighandler)

atexit.register(exithandler)

if len(sys.argv) > 1:
    subprocess.call(sys.argv[1:])
sys.exit(0)
