# == Define: twitch_haproxy::instance
#
# Setup haproxy instance pid/init/conf/service
#
# === Parameters
#
# [*svc_name*]
#  Name of haproxy instance, Default: ''
#
# [*defaults_options*]
#  Hash of options that creates defaults section of config, Default: {}
#
# [*global_options*]
#  Hash of options that creates global section of config, Default: {}
#
# [*haproxy_maxconn*]
#  Haproxy max connections, Default: 32000
#
# [*stats_port*]
#  Listen port for stats listener, Default: 2001
#
# [*stats_listen_address*]
#  Listen address for stats listener, Default: 0.0.0.0
#
# [*stats_options*]
#  Hash of settings for stats listener, Default: see below
define twitch_haproxy::instance(
  $svc_name             = '',
  $defaults_options     = {},
  $global_options       = {},
  $haproxy_maxconn      = 32000,
  $stats_port           = 2001,
  $stats_listen_address = '0.0.0.0',
  $stats_options        = {
    'mode'       => 'http',
    'stats uri'  => '/stats',
    'stats auth' => 'oxygen:atom',
    'option'     => [
      'httplog'
    ],
    'timeout'    => [
      'client 5000',
      'connect 500',
      'server 30000'
    ]
  },
) {
  include twitch_haproxy

  validate_hash($defaults_options)
  validate_hash($global_options)
  validate_hash($stats_options)

  $config_target = "/etc/haproxy/haproxy-${svc_name}.conf"

  twitch_haproxy::global{ $svc_name:
    defaults_options => $defaults_options,
    global_options   => $global_options,
    config_target    => $config_target,
    notify           => Service["haproxy_${svc_name}"],
    require          => Class['twitch_haproxy']
  }

  twitch_haproxy::listen{ "stats-${svc_name}":
    config_target       => $config_target,
    options             => $stats_options,
    ports               => $stats_port,
    ipaddress           => $stats_listen_address,
    notify              => Service["haproxy_${svc_name}"],
    default_remote_port => 2001,
  }

  service { "haproxy_${svc_name}":
    ensure     => running,
    enable     => true,
    hasrestart => true,
    hasstatus  => true,
    subscribe  => Package['haproxy'],
    require    => [
      Class['twitch_haproxy'],
      File["/etc/init.d/haproxy_${svc_name}"],
      Twitch_haproxy::Global[$svc_name],
    ],
  }

  file { "/etc/init.d/haproxy_${svc_name}":
    ensure  => present,
    owner   => root,
    group   => root,
    mode    => '0755',
    content => template('twitch_haproxy/haproxy-init.erb'),
  }
}
