# == Define Resource Type: twitch_haproxy::listen
#
# This type will setup a listening service configuration block inside
#  the $config_target file on an haproxy load balancer. Each listening service
#  configuration needs one or more load balancer member server (that should be
#  pass in via the $server_list param as an array).
#
# === Requirement/Dependencies:
#
# Currently requires the ripienaar/concat module on the Puppet Forge
#
# === Parameters
#
# [*name*]
#   The namevar of the defined resource type is the listening service's name.
#    This name goes right after the 'listen' statement in haproxy.cfg
#
# [*ports*]
#   Ports on which the proxy will listen for connections on the ip address
#    specified in the ipaddress parameter. Accepts either a single
#    comma-separated string or an array of strings which may be ports or
#    hyphenated port ranges.
#
# [*ipaddress*]
#   The ip address the proxy binds to. Empty addresses, '*', and '0.0.0.0'
#    mean that the proxy listens to all valid addresses on the system.
#
# [*options*]
#   A hash of options that are inserted into the listening service
#    configuration block.  This hash will write key / value pairs
#    to the configuration file.  For single world parameters, provide a
#    value of '' (ex. daemon).  For keys that are not unique (ex. option),
#    supply an array of parameters.
#
# [*bind_options*]
#   An array of options that will append to end of bind lines
#
# [*server_options*]
#   A string of options that will be appended to every server from $server_list
#
# [*config_target*]
#   The full path to the haproxy config to write.  Allows for multiple haproxy
#    instances on a single machine.  Default: /etc/haproxy/haproxy.cfg
#
# [*listen_template*]
#   Override of the default template to write listen blocks
#
# [*server_template*]
#   Override of the default template to write server lines with listen blocks.
#
# [*server_content*]
#   Override of the server config entries. If used, no server template is used
#   and the caller should generate the proper config entries.
#
# [*server_list*]
#   An array of strings or hashes (can not combine).
#     Strings: The template will parse the string for a ':'.  If one exists,
#       then that port value will be used.  If there is not a ':', the $default_remote_port
#       will be used for port.  The value will also look for a *space*.  Any content
#       will be appended to the server line for only that server.  This provides the
#       ability to have per machine configuration.
#       ex. ['app1.twitch.tv', # would use $default_remote_port and $server_options
#           'app2.twitch.tv:9999', # would use 9999 for port and $server_options
#           'app3.twitch.tv:9999 backup', # would use 9999 for port and append $server_options and 'backup'
#           'app4.twitch.tv backup'] # would use $default_remote_port for port and append $server_options and 'backup'
#
#     Hashes:  The template will look for the following keys: 'port','node','options'.
#     ex. [{'node' => 'app1.twitch.tv'}, # would use value of $default_remote_port for port and append $server_options
#          {'node' => 'app2.twitch.tv', 'port' => '9999' }, would use 9999 for port and append $server_options
#          {'node' => 'app3.twitch.tv', 'port' => '9999', 'options' => ['backup']}, would use 9999 for port and append $server_options and join the values from the 'options' key
#          {'node' => 'app3.twitch.tv','options' => ['backup']}, would use $default_remote_port for port and append $server_options and join the values from the 'options' key
#
# === Authors
#
# Gary Larizza <gary@puppetlabs.com>
# Changes from puppetlabs-haproxy::listen
# - 'mode' removed as define option, can be handled with other $options
# - added 'config_target' define option, in case multiple haproxy instances are needed
# - changed default values in the options define param
# - added 'server_content' so that custom server entries can be used
define twitch_haproxy::listen (
  $ports,
  $default_remote_port,
  $ipaddress        = [$::ipaddress],
  $config_target    = '/etc/haproxy/haproxy.cfg',
  $options          = {
    'option'  => [
      'forceclose',
      'httplog',
      'dontlog-normal',
      'dontlognull',
      'http-server-close'
    ],
    'balance' => 'roundrobin',
    'mode'    => 'http'
  },
  $bind_options     = [],
  $listen_template  = 'twitch_haproxy/haproxy_listen_block.erb',
  $server_template  = 'twitch_haproxy/haproxy_listen_block_servers.erb',
  $server_list      = [],
  $server_options   = '',
  $server_content   = '',
) {

  validate_absolute_path($config_target)
  validate_hash($options)
  validate_array($bind_options)
  validate_array($server_list)
  validate_string($server_options)

  # Template uses: $name, $ipaddress, $ports, $options
  concat::fragment { "${name}_listen_block":
    order   => "20-${name}-00",
    target  => $config_target,
    content => template($listen_template),
  }

  $server_contents = $server_content ? {
    ''      => template($server_template),
    default => $server_content,
  }

  concat::fragment { "${name}_listen_block_servers":
    order   => "20-${name}-01",
    target  => $config_target,
    content => $server_contents,
  }
}
