require 'rails_helper'

describe Api::BuildController, :type => :controller do

  describe 'POST create' do

    it 'creates a new build if valid' do
      expect(SmocaBuild.count).to eq(0)

      post :create, :developer => 'test', :job_name => 'qa-test', :initiated_at => Time.now
      expect(response).to have_http_status 200
      expect(SmocaBuild.count).to eq(1)
    end

    it 'returns a 400 if there was a failure' do
      expect(SmocaBuild.count).to eq(0)

      post :create
      expect(response).to have_http_status 400
      expect(SmocaBuild.count).to eq(0)
    end
  end

  describe 'PUT update' do

    describe 'can update fields' do

      it 'result' do
        record = FactoryGirl.create(:smoca_build, result: 'failure')
        expect(record.result).to eq('failure')

        put :update, :id => record.id, :result => 'success'
        record.reload

        expect(record.result).to eq('success')
      end

      it 'developer' do
        record = FactoryGirl.create(:smoca_build, developer: 'foo')
        expect(record.developer).to eq('foo')

        put :update, :id => record.id, :developer => 'foo'
        record.reload

        expect(record.developer).to eq('foo')
      end

      it 'environment' do
        record = FactoryGirl.create(:smoca_build, environment: 'staging-foo')
        expect(record.environment).to eq('staging-foo')

        put :update, :id => record.id, :environment => 'staging-bar'
        record.reload

        expect(record.environment).to eq('staging-bar')
      end

      it 'initiated_at' do
        first_time = Time.new(2010, 10, 31)
        second_time = Time.new(2012, 5, 25)

        record = FactoryGirl.create(:smoca_build, initiated_at: first_time)
        expect(record.initiated_at).to eq(first_time)

        put :update, :id => record.id, :initiated_at => second_time
        record.reload

        expect(record.initiated_at).to eq(second_time)
      end

      it 'build_id' do
        record = FactoryGirl.create(:smoca_build, build_id: '15')

        expect(record.build_id).to eq('15')

        put :update, :id => record.id, :build_id => '20'
        record.reload
        expect(record.build_id).to eq('20')
      end

      it 'job_name' do
        record = FactoryGirl.create(:smoca_build, job_name: 'foo')

        expect(record.job_name).to eq('foo')

        put :update, :id => record.id, :job_name => 'bar'
        record.reload
        expect(record.job_name).to eq('bar')
      end

      it 'queue_id' do
        record = FactoryGirl.create(:smoca_build, queue_id: '20')

        expect(record.queue_id).to eq('20')

        put :update, :id => record.id, :queue_id => '30'
        record.reload
        expect(record.queue_id).to eq('30')
      end

      it 'pending' do
        record = FactoryGirl.create(:smoca_build, pending: true)
        expect(record.pending).to eq(true)

        put :update, :id => record.id, :pending => false
        record.reload

        expect(record.pending).to eq(false)
      end

    end

    it 'can update multiple columns at once' do
      record = FactoryGirl.create(:smoca_build, pending: true, result: 'pending')

      expect(record.pending).to eq(true)
      expect(record.result).to eq('pending')

      put :update, :id => record.id, :pending => false, :result => 'failure'

      record.reload
      expect(record.pending).to eq(false)
      expect(record.result).to eq('failure')

    end

    it 'fails if one of multiple are invalid' do
      record = FactoryGirl.create(:smoca_build)

      put :update, :id => record.id, :pending => false, :job_name => nil

      expect(response).to have_http_status 400
    end

    it 'returns 200 when successful' do
      record1 = FactoryGirl.create(:smoca_build, :developer => 'foo')

      original_dev = record1.developer

      put :update, :id => record1.id, :developer => 'bar'
      record1.reload

      expect(record1.developer).not_to eq(original_dev)
      expect(response).to be_successful
      expect(response).to have_http_status 200
    end

    it 'returns a 404 error code if the ID does not exist' do
      put :update, :id => 38838538, :developer => 'LetsTryThis'

      expect(response).not_to be_successful
      expect(response).to have_http_status 404
    end
  end
end
