require 'rails_helper'

feature 'Dashboard Authentication', :type => :feature do

  scenario 'is not accessible when logged out' do
    visit dashboard_path

    expect(page).not_to have_current_path(dashboard_path)
    expect(page).not_to have_css('#dashboard_builds')
  end

  scenario 'is accessible when logged in' do
    log_into_dashboard

    expect(page).to have_current_path dashboard_path
    expect(page).to have_css('#dashboard_builds')
  end
end

feature 'Dashboard Past Builds', :type => :feature do

  scenario 'displays "No Builds Found" when there are none' do
    log_into_dashboard

    within('#dashboard_builds') do
      expect(page).to have_content 'No Builds Found'
    end
  end

  scenario 'displays builds when they exist' do
    create_builds

    log_into_dashboard

    within('#dashboard_builds') do
      expect(page).not_to have_content 'No Builds Found'

      expect(page).to have_css('#past_builds')
      within('#past_builds') do
        expect(page).to have_css('.build-item', count: 10) # Should have 10 builds
      end
    end
  end

  scenario 'other users builds do not appear on your dashboard' do
    create_builds

    log_into_dashboard('not_foobar')

    within('#past_builds') do
      expect(page).to have_content 'No Builds Found'
    end
  end

end

feature 'Dashboard Pending Builds', :type => :feature do

  scenario 'does not show when there are none' do
    log_into_dashboard

    within('#dashboard_builds') do
      expect(page).not_to have_css '#pending_builds'
      expect(page).not_to have_content 'In Progress Builds'
    end
  end

  scenario 'displays when there are some' do
    create_builds
    create_builds(1, true)

    log_into_dashboard

    within('#dashboard_builds') do
      expect(page).to have_css('#pending_builds')
      expect(page).to have_content('In Progress Builds')
    end

    within('#pending_builds') do
      expect(page).to have_css('.build-item', count: 1)
    end

    within('#past_builds') do
      expect(page).to have_css('.build-item', count: 10)
    end
  end

  scenario 'displays result as PENDING' do
    create_builds(10, true)

    log_into_dashboard

    within('#pending_builds') do
      build = first('.build-item')
      within build do
        expect(page).to have_css('#result', text: 'PENDING')
      end
    end
  end

  scenario 'do not appear in past builds section' do
    create_builds
    create_builds(1, true)

    log_into_dashboard

    within('#past_builds') do
      expect(page).not_to have_css('#result', text: 'PENDING')
    end
  end
end

# Creates builds in the database for testing
# @param count Int How many builds to create, default to 10
# @param pending Boolean Whether the build is pending, default to false
def create_builds(count= 10, pending= false)
  for i in 1..count
    FactoryGirl.create(:smoca_build,
                       developer: 'foobar',
                       result: (rand(1..2) == 1 ? 'SUCCESS' : 'FAILURE'),
                       pending: pending)

  end
end

# Shortcut to log into the dasboard
# @param username String Username to log in as
def log_into_dashboard(username='foobar')
  log_in_as(username, capybara: true)
  visit dashboard_path
end
