require 'rails_helper'

describe SmocaBuild do
  it 'has a valid factory' do
    expect(FactoryGirl.create(:smoca_build)).to be_valid
  end

  describe 'id' do
    it 'must be unique' do
      build_id = 1 # Both records will use the same id
      FactoryGirl.create(:smoca_build, id: build_id) # Original Build
      non_unique_record = FactoryGirl.build(:smoca_build, id: build_id) # Non-unique Build sharing id - 1
      expect(non_unique_record).not_to be_valid
    end
  end

  describe 'timestamps' do
    subject { build.valid? }

    context 'when not included' do
      let(:build) { FactoryGirl.build(:smoca_build, initiated_at: nil) }
      it { should be_falsey }
    end

    context 'when not in a date format' do
      let(:build) { FactoryGirl.build(:smoca_build, initiated_at: 'test') }
      it { should be_falsey }
    end
  end

  describe 'developers' do
    subject { build.valid? }

    context 'when not included' do
      let(:build) { FactoryGirl.build(:smoca_build, developer: nil) }
      it {should be_falsey}
    end

    context 'when blank' do
      let(:build) { FactoryGirl.build(:smoca_build, developer: '')}
      it {should be_falsey}
    end

    it "doesn't need to be unique" do
      developer = 'foobar'
      FactoryGirl.create(:smoca_build, developer: developer, id: 1, build_id: 1)

      build2 = FactoryGirl.build(:smoca_build, developer: developer, id: 2, build_id: 2)
      expect(build2).to be_valid
    end
  end

  describe 'job_name' do
    subject { build.valid? }

    context 'when not included' do
      let(:build) { FactoryGirl.build(:smoca_build, job_name: nil) }
      it { should be_falsey }
    end

    context 'when included' do
      let (:build) { FactoryGirl.build(:smoca_build, job_name: 'qa-smoca') }
      it { should be_truthy }
    end

    it "doesn't need to be unique" do
      job_type = 'qa-smoca'
      FactoryGirl.create(:smoca_build, job_name: job_type, build_id: 1)

      build_2 = FactoryGirl.build(:smoca_build, job_name: job_type, build_id: 2)
      expect(build_2).to be_valid
    end
  end

  describe 'build_id' do
    subject { build.valid? }

    context 'when not included' do
      let(:build) { FactoryGirl.build(:smoca_build, build_id: nil)}
      it { should be_truthy }
    end

    context 'when included' do
      let(:build) { FactoryGirl.build(:smoca_build, build_id: 5)}
      it { should be_truthy }
    end

    it 'can be duplicated with different job names' do
      FactoryGirl.create(:smoca_build, job_name: 'qa-smoca', build_id: 1)
      build2 = FactoryGirl.build(:smoca_build, job_name: 'qa-video', build_id: 1)

      expect(build2).to be_valid
    end

    it 'can not be duplicated with same job name' do
      job_name = 'qa-test'
      build_id = 1

      FactoryGirl.create(:smoca_build, job_name: job_name, build_id: build_id)
      build2 = FactoryGirl.build(:smoca_build, job_name: job_name, build_id: build_id)

      expect(build2).not_to be_valid
    end

    it 'can be duplicated if nil' do
      # Use case, if a user created a new build via /smoca_build/new, but it hasn't assigned a build_id yet
      job_name = 'qa-test'

      FactoryGirl.create(:smoca_build, job_name: job_name, build_id: nil)
      build2 = FactoryGirl.build(:smoca_build, job_name: job_name, build_id: nil)
      expect(build2).to be_valid
    end

    it 'can be duplicated if blank' do
      # Use case, if a user created a new build via /smoca_build/new, but it hasn't assigned a build_id yet
      job_name = 'qa-test'

      FactoryGirl.create(:smoca_build, job_name: job_name, build_id: '')
      build2 = FactoryGirl.build(:smoca_build, job_name: job_name, build_id: '')
      expect(build2).to be_valid
    end
  end

  describe 'name method' do
    let(:build) { FactoryGirl.create(:smoca_build, build_id: 5)}

    it 'returns a proper name' do
      expect(build.name).to match('qa-smoca #5')
    end

  end

  describe 'jenkins_log_url method' do
    let(:job_name) { 'qa-smoca' }
    let(:build_id) { 5 }

    it 'returns a url if necessary build_id and job_name are provided' do
      build = FactoryGirl.create(:smoca_build, build_id: build_id, job_name: job_name)
      expect(build.jenkins_log_url).to eq("https://jenkins.internal.justin.tv/job/#{job_name}/#{build_id}/console")
    end

    it 'returns nil if build_id is not provided' do
      build = FactoryGirl.create(:smoca_build, build_id: nil, job_name: job_name)
      expect(build.jenkins_log_url).to be_nil
    end
  end
end
