require 'rails_helper'

describe TestCaseResult, type: :model do
  it 'has a valid factory' do
    expect(FactoryGirl.create(:test_case_result)).to be_valid
  end

  describe 'result' do
    subject { record.valid? }

    context 'when included' do
      let(:record) { FactoryGirl.build(:test_case_result, result: 'success') }
      it { should be_truthy }
    end

    context 'when not included' do
      let(:record) { FactoryGirl.build(:test_case_result, result: nil) }
      it { should be_falsey }
    end

     context 'when blank' do
       let(:record) { FactoryGirl.build(:test_case_result, result: '   ') }
       it { should be_falsey }
     end

    it 'does not need to be unique' do
      result = 'success'
      record1 = FactoryGirl.create(:test_case_result, result: result)

      record2 = FactoryGirl.build(:test_case_result, result: result)
      expect(record2).to be_valid
    end
  end

  describe 'exception_class' do
    subject { record.valid? }

    context 'when included' do
      let(:record) { FactoryGirl.build(:test_case_result, exception_class: 'Capybara::ElementNotFound') }
      it { should be_truthy }
    end

    context 'when not included' do
      let(:record) { FactoryGirl.build(:test_case_result, exception_class: nil, exception_message: nil) }
      it { should be_truthy }
    end

    context 'when blank' do
      let(:record) { FactoryGirl.build(:test_case_result, exception_class: '   ', exception_message: nil) }
      it { should be_truthy }
    end

    it 'does not need to be unique' do
      exception_class = 'Capybara::ElementNotFound'
      record1 = FactoryGirl.create(:test_case_result, exception_class: exception_class)

      record2 = FactoryGirl.build(:test_case_result, exception_class: exception_class)
      expect(record2).to be_valid
    end
  end

  describe 'exception_message' do
    subject { record.valid? }

    context 'when included' do
      let(:record) { FactoryGirl.build(:test_case_result, exception_message: 'Unable to find css ".ember-chat"') }
      it { should be_truthy }
    end

    context 'when not included' do
      let(:record) { FactoryGirl.build(:test_case_result, exception_message: nil) }
      it { should be_truthy }
    end

    context 'when blank' do
      let(:record) { FactoryGirl.build(:test_case_result, exception_message: '   ') }
      it { should be_truthy }
    end

    it 'when long' do
      b = FactoryGirl.create(:test_case_result, exception_message: 'a' * 2000)
      expect(b).to be_valid
    end

    it 'does not need to be unique' do
      exception_message = 'Unable to find css ".ember-chat"'
      record1 = FactoryGirl.create(:test_case_result, exception_message: exception_message)

      record2 = FactoryGirl.build(:test_case_result, exception_message: exception_message)
      expect(record2).to be_valid
    end

    context 'when missing an exception class' do
      let(:record) { FactoryGirl.build(:test_case_result, exception_class: nil, exception_message: 'Foobar') }
      it { should be_falsey }
    end
  end

  describe 'failing_line' do
    failing_line = "Failure/Error: expect(page).to have_css '.gejgiejie' "
    subject { record.valid? }

    context 'when included' do
      let(:record) { FactoryGirl.build(:test_case_result, failing_line: failing_line) }
      it { should be_truthy }
    end

    context 'when not included' do
      let(:record) { FactoryGirl.build(:test_case_result, failing_line: nil) }
      it { should be_truthy }
    end

    it 'does not need to be unique' do
      FactoryGirl.create(:test_case_result, failing_line: failing_line)
      build = FactoryGirl.build(:test_case_result, failing_line: failing_line)
      expect(build).to be_valid
    end
  end

  describe 'backtrace' do
    backtrace = "./spec/test_spec.rb:6:in `block (2 levels) in <top (required)>'\\n./core/base/spec_helper.rb:127:in `block (2 levels) in <top (required)>'"
    subject { record.valid? }

    context 'when included' do
      let(:record) { FactoryGirl.build(:test_case_result, backtrace: backtrace) }
      it { should be_truthy }
    end

    context 'when not included' do
      let(:record) { FactoryGirl.build(:test_case_result, backtrace: nil) }
      it { should be_truthy }
    end

    it 'does not need to be unique' do
      FactoryGirl.create(:test_case_result, backtrace: backtrace)
      build = FactoryGirl.create(:test_case_result, backtrace: backtrace)
      expect(build).to be_valid
    end

    it 'can be up to 65535 characters' do
      length_limit = 65535
      b = FactoryGirl.build(:test_case_result, backtrace: 'a' * length_limit)
      expect(b).to be_valid

      c = FactoryGirl.build(:test_case_result, backtrace: 'a' * (length_limit + 1))
      expect(c).not_to be_valid
    end
  end

  describe 'screenshot_url' do
    screenshot_url = "https://google.com"
    subject { record.valid? }

    context 'when included' do
      let(:record) { FactoryGirl.build(:test_case_result, screenshot_url: screenshot_url) }
      it { should be_truthy }
    end

    context 'when not included' do
      let(:record) { FactoryGirl.build(:test_case_result, screenshot_url: nil) }
      it { should be_truthy }
    end

    context 'when long' do
      let(:record) { FactoryGirl.build(:test_case_result, screenshot_url: screenshot_url * 2000) }
      it { should be_falsey }
    end

    it 'does not need to be unique' do
      FactoryGirl.create(:test_case_result, screenshot_url: screenshot_url)
      build = FactoryGirl.create(:test_case_result, screenshot_url: screenshot_url)
      expect(build).to be_valid
    end
  end
end
