require 'rails_helper'

describe User, type: :model do
  it 'has a valid factory' do
    expect(FactoryGirl.create(:user)).to be_valid
  end

  describe 'admin status' do
    subject { user.admin? }

    context 'when not specified' do
      let(:user) { FactoryGirl.create(:user, admin: '') }
      it { should be_falsey }
    end

    context 'when specified as false' do
      let(:user) { FactoryGirl.create(:user, admin: false) }
      it { should be_falsey }
    end

    context 'when specified as true' do
      let(:user) { FactoryGirl.create(:user, admin: true) }
      it { should be_truthy }
    end
  end

  describe 'uid' do
    subject { user.valid? }

    context 'when not provided' do
      let(:user) { FactoryGirl.build(:user, uid: '') }
      it { should be_falsey }
    end

    context 'when provided' do
      let(:user) { FactoryGirl.build(:user, uid: 'foo') }
      it { should be_truthy }
    end

    context 'when only white space' do
      let(:user) { FactoryGirl.build(:user, uid: '   ')}
      it { should be_falsey }
    end
  end

  describe 'provider' do
    subject { user.valid? }

    context 'when not provided' do
      let(:user) { FactoryGirl.build(:user, provider: '') }
      it { should be_falsey }
    end

    context 'when provided' do
      let(:user) { FactoryGirl.build(:user, provider: 'foo') }
      it { should be_truthy }
    end

    context 'when only white space' do
      let(:user) { FactoryGirl.build(:user, provider: '   ')}
      it { should be_falsey }
    end
  end

  it 'must be a unique uid' do
    user = FactoryGirl.create(:user, uid: 'foobar')
    user2 = FactoryGirl.build(:user, uid: 'foobar')

    expect(user2).not_to be_valid
  end

  it "provider doesn't need to be unique" do
    user = FactoryGirl.create(:user, uid: 'user1', provider: 'ldap')
    user2 = FactoryGirl.create(:user, uid: 'user2', provider: 'ldap')

    expect(user2).to be_valid
  end

  it "email doesn't need to be unique" do
    user = FactoryGirl.create(:user, uid: 'user1', email: 'foo@bar.com')
    user2 = FactoryGirl.build(:user, uid: 'user2', email: 'foo@bar.com')

    expect(user2).to be_valid
  end
end
