require_relative 'environment_config'
require_relative 'passport_environment'
require_relative 'kraken_environment'

class WebEnvironment < EnvironmentConfig

  class << self

    attr_reader :videoshim_host, :url

    # Configures Web Environment and stores the URL
    # Includes initialization of dependencies for KrakenEnvironment and PassportEnvironment
    # @param [String] environment The environment name
    def initialize(environment_input)

      # Standardize the user input for environment. Downcase it, parse clean- from it.
      environment = environment_input.downcase if environment_input.is_a? String
      environment = parse_clean(environment)

      @url = host_url(environment)

      case environment
        when 'local'
          KrakenEnvironment.initialize('local')
        when 'production', 'production-canary'
          PassportEnvironment.initialize('production')
          KrakenEnvironment.initialize('production')
          @videoshim_host = 'http://video-shim.internal.twitch.tv:5151'
        when nil
          puts warning_message('WARNING: No ENVIRONMENT variable was provided. We suggest you provide one. Defaulting to production.')
          KrakenEnvironment.initialize('production')
          PassportEnvironment.initialize('production')
          @videoshim_host = 'http://video-shim.internal.twitch.tv:5151'
          ENV['ENVIRONMENT'] = 'production'
        else
          KrakenEnvironment.initialize(environment)
          @videoshim_host = 'http://videoshim.dev.us-west2.twitch.tv'
          PassportEnvironment.initialize('staging')
      end
    end

    # This method does not maintain state.
    # Use initialize() to set an environment and refer to be able to refer to the FQDN elsewhere.
    # @param [String] environment The environment name
    # @return [String] The fully qualified domain name of the specified environment
    def host_url(environment)
      case environment
        when 'local'
          return local_host
        when 'production', 'production-canary'
          return production_host
        when nil
          return production_host
        else
          return staging_host(environment)
      end
    end

    # Removes any clean- prefix from an environment. This is specified by Skadi, but isn't used in the URL
    # @param [String] environment The environment
    # @return [String] An environment, with clean- parsed if applicable
    private def parse_clean(environment)
      if environment =~ /clean-(.+)/
        return $1
      else
        return environment
      end
    end

    # @return [String] Web localhost fully qualified domain name
    private def local_host
      return 'https://localhost.twitch.tv:4200'
    end

    # @return [String] Web production fully qualified domain name
    private def production_host
      return 'https://www.twitch.tv'
    end

    # @param [String] environment The staging environment
    # @return [String] Staging fully qualified domain name
    private def staging_host(environment)
      return "https://#{environment}-www.dev.us-west2.twitch.tv"
    end

  end


end
