require 'json'
require 'fileutils'
require './core/utils/stats_utils'

module PostBuildUtils

  include StatUtils

  # Rake Post-Build Methods

  # Calling this will call other methods to output a Post-Build Summary
  def output_summary
    print_duration
    puts "\n"
    print_login_results
    puts "\n\n"
    print_pending_output
    puts "\n\n"
    print_failure_output
  end

  # Output the Pending Scenarios
  def print_pending_output
    pending = get_pending

    if pending.length > 0
      puts "Total Scenarios Pending: #{pending.length}\n\n"
      pending.each_with_index do |pending, index|
        puts "#{index.next}) #{pending}\n\n"
      end
    end
  end

  # Output the Process Durations
  # TODO: Track the duration to statsd
  def print_duration
    durations = get_test_duration
    durations.each_with_index do |duration, index|
      puts "Process #{index.next}: Duration: #{Time.at(duration.to_f).utc.strftime("%H:%M:%S")}" if durations.length > 1
    end
  end

  # Output LoginPage Success/Failure Results
  def print_login_results
    login_success = get_login_success
    login_fail = get_login_failure

    if login_success.length > 0 || login_fail.length > 0
      logins = StatsCounter.new

      # Iterate through the login_success and login_fail arrays. Take the value and log it with the StatsCounter object
      # It's important to do this so that we can access percentage methods
      login_success.each do |x|
        logins.success += x.to_f
      end

      login_fail.each do |y|
        logins.fail += y.to_f
      end

      logins.calculate_percents # Manually call this to recalculate the percent (since we incremented manually)
      puts "LoginPage Success Count: #{logins.success.to_i}\n"\
    "LoginPage Fail Count: #{logins.fail.to_i}\n"\
    "Percentage of Success: #{logins.success_percent}%"
      # Printing out success percent instead of failure because in the logs it looks better if 100%
      # But in terms of statsd graphs, it's only important to notice failure rates increase
      # Easily changeable to fail_percentage if needed (or simply removing the print statement)

      gauge('debug.login_failure.count', logins.fail_percent) # Send login fail stats if tracking enabled
      # TODO: Pull this out of the Print Method. It's unclear that this method is used for tracking as well.
    end
  end

  # Output the Test Failures
  def print_failure_output
    failures = get_failures

    if failures.length > 0
      puts "Total Failures: #{failures.length}\n\n"

      failures.each_with_index do |failure, index|
        puts "#{index.next}) #{failure}\n\n"
      end
    else
      puts 'No Failures'
    end
  end

  private

  # Takes JSON log data and writes it to a file
  # @param filename String The file name to write to
  # @param data The data to write to the file
  def write_logs_to_file(filename, data)
    path = "#{failure_location_path}/browser_logs/#{filename}"

    # Create directory if path not yet created
    dirname = File.dirname(path)
    FileUtils.mkdir_p(dirname) unless File.directory?(dirname)

    File.open(path, 'w') { |f| f.write(data) }
    return path
  end

end
