require './core/base/spec_helper'
require './core/configs/bmp_config'
require './core/utils/console_prints_utils'
require './core/network_traffic_analytics/video_player/tier1_event_filtering'
require './core/api/video_player/stream_metadata_api'
require './core/twitch/video_player/player_page'
require './core/network_traffic_analytics/filtering'

include ConsolePrintsUtils

player_url = VideoPlayerData.player_stream_url
player_domain = VideoPlayerData.player_domain
player_channel = VideoPlayerData.player_channel
player_host = VideoPlayerData.player_host
video_play_schema = '/video_player/tier1_events/video_play_event_schema'
minute_watched_schema = '/video_player/tier1_events/m_w_event_schema'
buffer_empty_schema = '/video_player/tier1_events/buffer_empty_schema'
buffer_refill_schema = '/video_player/tier1_events/buffer_refill_schema'

feature 'Validating Network Tracked Tier 1 Events:' do
  let(:tier1) { Tier1EventFiltering.new }
  let(:player) { PlayerPage.new('stream') }
  let(:network_traffic) { Filtering.new(get_proxy) }

  before(:all) do
    @proxy = get_proxy
    StreamMetadataApi.get_stream_metadata(player_channel)
  end

  before(:each) do
    @proxy.new_har
    PlayerPage.visit_with_localstore_settings(player_url)

    if VideoPlayerData.player_type == 'flash'
      # Flash is buggy. Need to set channel after the localstore setting change
      # See Jira Ticket VP-3005
      PlayerPage.set_channel(VideoPlayerData.player_channel)
    end

    PlayerPage.display_player_stats # Debugging hanging browser/proxy issues. See SMOC-337.
  end

  scenario "'video-play' fired upon stream start.", :unsupported_browsers => [:safari] do
    expected_event = {
      'video-play' => [true, video_play_schema, player_url, player_host, player_domain, player_channel]
    }

    sleep(20) # Simulating video player viewing up to 20 seconds waiting for video-play event.

    spade_events = network_traffic.filter_spade_events
    event = tier1.get_video_play_event(spade_events)

    begin
      expect(event).to eq(expected_event)
    rescue RSpec::Expectations::ExpectationNotMetError => e
      puts "Encountered error #{e.class}.\nSpade Events: #{spade_events}"
      TwitchUtils.debug_logging(network_traffic.get_json_har, "video_play")
      raise e
    end
  end

  scenario "'minute-watched' fired twice in 2 un-interrupted mins upon steam start.", :unsupported_browsers => [:safari] do
    expected_event = {
      'minute-watched' => [true, minute_watched_schema, player_url, player_host, player_domain, player_channel, 1],
      'm_w_2' => [true, minute_watched_schema, player_url, player_host, player_domain, player_channel, 2]
    }

    # Simulating video player viewing up to 120 seconds waiting for two minutes-watched events.
    # Adding 5 seconds to the wait in case player loaded slowly.
    TwitchUtils.fluent_wait(30, 120)
    spade_events = network_traffic.filter_spade_events
    event = tier1.get_minute_watched_event(spade_events)

    begin
      expect(event).to eq(expected_event)
    rescue RSpec::Expectations::ExpectationNotMetError => e
      puts "Encountered error #{e.class}.\nSpade Events: #{spade_events}"
      TwitchUtils.debug_logging(network_traffic.get_json_har, "minute_watched")
      raise e
    end
  end

  scenario "'buffer-empty' fired after limiting bandwidth upon landing on player.", :tier1_buffer_events => :true do
    expected_event = {
      'buffer-empty' => [true, buffer_empty_schema, player_url, player_host, player_domain, player_channel]
    }

    sleep(30) # Waiting 30 seconds for the video to load up and ad to finish before limiting the bandwidth.

    @proxy.limit_proxy_bandwidth(5000, 250, 50) # Adding 5 seconds latency, 250kbs downstream, 50kbs upstream.

    sleep(30) # Waiting 30 seconds for the video hit buffer-empty

    spade_events = network_traffic.filter_spade_events
    event = tier1.get_buffer_empty_event(spade_events)

    begin
      expect(event).to eq(expected_event)
    rescue RSpec::Expectations::ExpectationNotMetError => e
      puts "Encountered error #{e.class}.\nSpade Events: #{spade_events}"
      TwitchUtils.debug_logging(network_traffic.get_json_har, "buffer_empty")
      raise e
    end

    @proxy.limit_proxy_bandwidth(0, 0, 0) # Remove added latency, downstreamKbps, and upstreamKpbs
  end

  scenario "'buffer-refill' fired after bandwith was limited and then lifted upon landing on player.", :tier1_buffer_events => :true do
    expected_event = {
      'buffer-refill' => [true, buffer_refill_schema, player_url, player_host, player_domain, player_channel]
    }

    sleep(30) # Waiting 30 seconds for the video to load up and ad to finish before limiting the bandwidth.

    @proxy.limit_proxy_bandwidth(5000, 0, 0) # Adding 5 seconds latency, 250kbs downstream, 50kbs upstream.
    sleep(30) # Waiting 30 seconds for the video hit buffer-empty and buffer-refill

    @proxy.limit_proxy_bandwidth(0, 0, 0) # Remove added latency, downstreamKbps, and upstreamKpbs

    sleep(30) # Waiting 30 seconds for the video to ensure buffer-refill

    spade_events = network_traffic.filter_spade_events
    event = tier1.get_buffer_refill_event(spade_events)

    begin
      expect(event).to eq(expected_event)
    rescue RSpec::Expectations::ExpectationNotMetError => e
      puts "Encountered error #{e.class}.\nSpade Events: #{spade_events}"
      TwitchUtils.debug_logging(network_traffic.get_json_har, "buffer_refill")
      raise e
    end
  end

end
