require './core/base/spec_helper'
require './core/twitch/web/passport/signup_page'
require './core/twitch/web/components/top_nav'

include SignupPage

feature 'Sign Up Page' do
  before(:all) do
    begin
      set_bypass_captcha_cookie
    rescue Selenium::WebDriver::Error::ScriptTimeoutError => e
      # Try one more time... temporary. QE-597. Remove this catch when QE-598 is complete
      logger.warn "Encountered [#{e.class}] - #{e.message}.\nRetrying..."
      set_bypass_captcha_cookie
    end
  end

  before(:each) do
    visit '/signup'

    if PageHelper.get_cookie(get_captcha_bypass_cookie_name_signup).nil?
      pre_url = current_url
      set_bypass_captcha_cookie
      visit pre_url # Refresh
    end
  end

  current_year = 2 # Sign Up Year Form: option:nth-child(2).
  # Assigning this so you can go current_year + 12 to indicate 12 years old
  fifteen_years = "option:nth-child(#{current_year + 15})" # The CSS of the birthday dropdown to make you 15 years old
  underage = "option:nth-child(#{current_year + 5})" # The CSS of the birthday dropdown to make you 5 years old
  default_birthdate_option = 'option:nth-child(4)'

  scenario 'displays sign up by default', :tier1 => true do
    expect(page).to have_css('#signup_tab.tab.selected')
  end

  scenario 'cannot sign up with all fields blank', :tier2 => true do
    expect(get_signup_button.disabled?).to eq(true) # Expect Sign Up button to be disabled
  end

  scenario 'cannot sign up for existing account', :tier2 => true do
    fill_in_fields(true,
      {
        :username => 'twitch_automation'
      })

    validator_expectation(false, true, true)
    expect(get_signup_button.disabled?).to eq(true)
  end

  scenario 'cannot sign up with a special character username', :tier2 => true do
    fill_in_fields(true,
      {
        :username => 'qa_auto&#'
      })

    validator_expectation(false, true, true)
    expect(get_signup_button.disabled?).to eq(true)
  end

  scenario 'cannot sign up with a username containing a space', :tier2 => true do
    fill_in_fields(true,
      {
        :username => 'qa auto'
      })

    validator_expectation(false, true, true)
    expect(get_signup_button.disabled?).to eq(true)
  end

  scenario 'cannot sign up for a username too long', :tier2 => true do
    max_length = 30
    username = 'qa_auto' << '9' * max_length

    fill_in_fields(true,
      {
        :username => username
      })

    validator_expectation(false, true, true)
    expect(get_signup_button.disabled?).to eq(true)
  end

  scenario 'cannot sign up without a password', :tier2 => true do
    fill_in_fields(true,
      {
        :password => ''
      })

    sleep(0.5) # Sleep to ensure a validator has enough time to display after the rapid input
    expect(get_signup_button.disabled?).to eq(true)
  end

  scenario 'cannot sign up with a weak password', :tier2 => true do
    fill_in_fields(true,
      {
        :password => '12345'
      })

    validator_expectation(true, false, true)
    expect(get_signup_button.disabled?).to eq(true)
  end

  scenario 'cannot sign up with an incomplete birthday', :tier2 => true do
    skip('Currently Broken. ID-203')
    fill_in_fields(false,
      {
        :username => generate_username,
        :password => 'L3tsTw1tch',
        :email => 'qa.automation@gmail.com'
      })

    sleep(0.5) # Sleep to ensure a validator has enough time to display after the rapid input
    expect(get_signup_button.disabled?).to eq(true)

    # Now try adding the month, no day or year
    fill_in_fields(false,
      {
        :month => default_birthdate_option
      })
    sleep(0.5) # Sleep to ensure a validator has enough time to display after the rapid input
    expect(get_signup_button.disabled?).to eq(true)

    # Now try adding the day, no year
    fill_in_fields(false,
      {
        :day => default_birthdate_option
      })

    sleep(0.5) # Sleep to ensure a validator has enough time to display after the rapid input
    expect(get_signup_button.disabled?).to eq(true)
  end

  scenario 'cannot sign up while under the age of 13', :tier2 => true do
    fill_in_fields(true,
      {
        :year => underage, # 5 years old
      })

    sleep(0.5)
    get_signup_button.click

    expect(find('.subwindow_notice').text).to eq('You are not eligible to create an account at this time.')

    # Make sure the user can't simply change the age to over 13 and register without clearing cookies/cache

    visit('/signup')

    fill_in_fields(true,
      {
        :year => fifteen_years, # 15 years old
      })

    sleep(0.5)
    get_signup_button.click

    expect(find('.subwindow_notice').text).to eq('You are not eligible to create an account at this time.') # Message should be cached
  end

  scenario 'cannot sign up without an email address', :tier2 => true do
    fill_in_fields(true,
      {
        :email => ''
      })

    sleep(0.5) # Sleep to ensure a validator has enough time to display after the rapid input
    expect(get_signup_button.disabled?).to eq(true)
  end


  scenario 'can sign up for a new account, lowercase', :tier2 => true do
    skip('This is designed for use against staging') if environment_production?

    username = generate_username.downcase # Assigning to a variable to compare later

    fill_in_fields(true,
      {
        :username => username
      })

    validator_expectation(true, true, true)

    button = get_signup_button
    expect(button.disabled?).to eq(false)
    button.click

    expect(page).not_to have_css('.subwindow_notice') # No failure
    # expect(page).to have_current_path('/', wait: 5) # Expect to redirect to front page TODO: Removed for ID-1577
    expect(TopNav.element).to have_content(/#{username}/i) # Look for the username, case insensitive
  end

  scenario 'can sign up for a new account, uppercase', :tier2 => true do
    skip('This is designed for use against staging') if environment_production?

    username = generate_username().upcase # Assigning to a variable to compare later

    fill_in_fields(true,
      {
        :username => username
      })

    validator_expectation(true, true, true)

    button = get_signup_button
    expect(button.disabled?).to eq(false)
    button.click

    expect(page).not_to have_css('.subwindow_notice') # No failure
    # expect(page).to have_current_path('/', wait: 5) # Expect to redirect to front page TODO: Removed for ID-1577
    expect(TopNav.element).to have_content(/#{username}/i) # Look for the username, case insensitive
  end
end

feature 'Sign Up Links' do
  before do
    visit '/signup'
  end

  scenario 'can click Terms of Service', :tier2 => true do
    skip('Legal endpoint has changed. Needs to be updated.')
    tos_window = window_opened_by do
      click_link 'Terms of Service'
    end

    within_window tos_window do
      # TODO: Figure out how to use current_path with this. current_path isn't including the parameters
      expect(current_url).to include('p/terms-of-service') # Cannot match, as staging Passport exists on staging-12
      # Wouldn't match Capybara.app_host
    end
  end

  scenario 'can click Privacy Policy', :tier2 => true do
    skip('Legal endpoint has changed. Needs to be updated.')
    privacy_window = window_opened_by do
      click_link 'Privacy Policy'
    end

    within_window privacy_window do
      # TODO: Figure out how to use current_path with this. current_path isn't including the parameters
      expect(current_url).to include('p/privacy-policy') # Cannot match, as staging Passport exists on staging-12
    end
  end
end

feature 'Sign Up Validators' do
  before do
    visit '/signup'
  end
  scenario "Don't appear by default", :tier2 => true do
    init_variables() # Get the validator_css vars
    expect(page).not_to have_css(@username_validator_css)
    expect(page).not_to have_css(@password_validator_css)
    expect(page).not_to have_css(@email_validator_css)
  end

  scenario 'properly update when user changes input', :tier2 => true do
    fill_in_fields(true,
      {
        :username => 'qa auto',
        :password => '12345',
        :email => 'blah blah blah'
      })

    validator_expectation(false, false, false)

    fill_in_fields(false,
      {
        :username => generate_username,
        :password => 'L3tsTw1tch',
        :email => 'qa.automation@gmail.com'
      })

    sleep(0.5)
    validator_expectation(true, true, true)
  end
end
