# Module that includes very basic support of an EC2 Machine.
# Also includes a Alarm Instance, that can be configured.

provider "aws" {
  region  = "us-west-2"
  profile = "twitch-web-dev"
}

variable owner {} # The owner of the instance
variable cluster {} # The cluster type. Valid clusters include selenium-node-firefox, selenium-node-chrome and selenium-hub
variable count { # The amount of instances to spin up
  default = "1"
}
variable node_instance {} # The size of the box. I.e., c3.large
variable max_cpu_threshold_alarm { # Max CPU Usage before monitoring fails, defaulted to 75%
  default = "75"
}

variable "ami" {
  default = "ami-af17decf" # Already configured for Puppet & Docker support
}

# Refer to twitch_web_dev to be able to get things like the security_groups
data "terraform_remote_state" "twitch_web_dev" {
  backend = "s3"
  config {
    profile = "twitch-web-dev"
    bucket = "twitch-web-dev"
    key = "tfstate/web/terraform/accounts/twitch-web-dev/terraform.tfstate"
  }
}

resource "aws_instance" "node" {
  ami = "${var.ami}"
  count = "${var.count}"
  instance_type = "${var.node_instance}"

  # Security/Subnet Groups for internal access only
  vpc_security_group_ids = ["${data.terraform_remote_state.twitch_web_dev.twitch_subnets_sg}"]
  subnet_id = "${data.terraform_remote_state.twitch_web_dev.service_a}"

  tags {
    Name = "${var.cluster}-${count.index+1}" # Ex: my-cluster-4
    Owner = "${var.owner}"
    Team = "CAPE/QE"
  }
}

output "private_ip" {
  value = "${aws_instance.node.private_ip}"
}

# Cloudwatch Configuration for the Node

# Remote State to be able to pull the SNS ARN for the alarm
data "terraform_remote_state" "smoca_sns" {
  backend = "s3"
  config {
    profile = "twitch-web-dev"
    bucket = "tf-state-qa"
    key = "tfstate/qa/smoca/account/twitch-web-dev/cloudwatch"
  }
}

resource "aws_cloudwatch_metric_alarm" "node-alarm" {
  count = "${var.count}"
  alarm_name = "${var.cluster}-cpu-usage-${element(aws_instance.node.*.id, count.index)}"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  evaluation_periods = "1"
  metric_name = "CPUUtilization"
  namespace = "AWS/EC2"
  period = "300" # 5 minutes
  statistic = "Average"
  threshold = "${var.max_cpu_threshold_alarm}"
  alarm_description = "This metric monitors the CPU Utilization of ${count.index+1}"
  dimensions = {InstanceId = "${element(aws_instance.node.*.id, count.index)}"}
  alarm_actions = ["${data.terraform_remote_state.smoca_sns.grid-monitoring-sns-arn}"]
  ok_actions = ["${data.terraform_remote_state.smoca_sns.grid-monitoring-sns-arn}"]
}
