# This module serves to create EC2 instances for Selenium Grid nodes.
# Variables of Owner, Cluster, Count and node_instance must be passed in

variable owner {} # The owner of the instance
variable cluster {} # The cluster type. Valid clusters include selenium-node-firefox, selenium-node-chrome and selenium-hub
variable count {} # The amount of instances to spin up
variable node_instance {} # The size of the box. I.e., c3.large
variable root_block_volume_size { # The size of the volume, in GBs
  default = "8"
}
variable max_cpu_threshold_alarm { # Max CPU Usage before monitoring fails, defaulted to 75%
  default = "75"
}

variable "ami" {
  default = "ami-af17decf" # Already configured for Puppet & Docker support
}

# Refer to twitch_web_dev to be able to get things like the security_groups
data "terraform_remote_state" "twitch_web_dev" {
  backend = "s3"
  config {
    profile = "twitch-web-dev"
    bucket = "twitch-web-dev"
    key = "tfstate/web/terraform/accounts/twitch-web-dev/terraform.tfstate"
  }
}

##
# AWS access keys must be provided within .bash_profile
##

provider "aws" {
  profile = "twitch-web-dev"
}

resource "aws_instance" "node" {
  ami = "${var.ami}"
  count = "${var.count}"
  instance_type = "${var.node_instance}"

  # Security/Subnet Groups for internal access only
  vpc_security_group_ids = ["${data.terraform_remote_state.twitch_web_dev.twitch_subnets_sg}"]
  subnet_id = "${data.terraform_remote_state.twitch_web_dev.service_a}"

  # Run puppet when starting
  # Set the hostname to something unique - using the instance ID
  user_data = <<END_OF_STRING
#cloud-config
preserve_hostname: true
bootcmd:
 - cloud-init-per instance my_set_hostname sh -xc "echo "${var.cluster}-$INSTANCE_ID" | sed -e 's/-i-/-/g' > /etc/hostname; hostname -F /etc/hostname"
 - cloud-init-per instance my_etc_hosts sh -xc "sed -i -e '/^127.0.1.1/d' /etc/hosts; echo 127.0.1.1 $(cat /etc/hostname).dev.us-west2.justin.tv $(cat /etc/hostname) >> /etc/hosts"
runcmd:
 - echo "cluster=${var.cluster}" > /etc/facter/facts.d/cluster.txt
 - echo "clean=true" > /etc/facter/facts.d/clean.txt
 - puppet agent --test --server=puppet.internal.justin.tv
END_OF_STRING

  tags {
    Name = "${var.cluster}-${count.index+1}" # Ex: selenium-node-firefox-4
    Owner = "${var.owner}"
    Team = "qa/smoca"
  }

  root_block_device {
    volume_size = "${var.root_block_volume_size}"
  }
}

output "private_ip" {
  value = "${aws_instance.node.private_ip}"
}

# Cloudwatch Configuration for the Node

# Remote State to be able to pull the SNS ARN for the alarm
data "terraform_remote_state" "smoca_sns" {
  backend = "s3"
  config {
    profile = "twitch-web-dev"
    bucket = "tf-state-qa"
    key = "tfstate/qa/smoca/account/twitch-web-dev/cloudwatch"
  }
}

resource "aws_cloudwatch_metric_alarm" "node-alarm" {
  count = "${var.count}"
  alarm_name = "${var.cluster}-cpu-usage-${element(aws_instance.node.*.id, count.index)}"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  evaluation_periods = "1"
  metric_name = "CPUUtilization"
  namespace = "AWS/EC2"
  period = "300" # 5 minutes
  statistic = "Average"
  threshold = "${var.max_cpu_threshold_alarm}"
  alarm_description = "This metric monitor grid ec2 cpu utilization"
  dimensions = {InstanceId = "${element(aws_instance.node.*.id, count.index)}"}
  alarm_actions = ["${data.terraform_remote_state.smoca_sns.grid-monitoring-sns-arn}"]
  ok_actions = ["${data.terraform_remote_state.smoca_sns.grid-monitoring-sns-arn}"]
}
