﻿using System;
using Newtonsoft.Json;
using System.Collections.Generic;
using System.Threading.Tasks;

namespace CBGModules
{
    /// <summary>
    /// Contains methods for interacting with the Central Syslog Server
    /// </summary>
    public static class Syslog
    {
        public static String endpoint = "http://oe-tools-team-syslog.prod.us-west2.justin.tv";

        /// <summary>
        /// Write the message and level to syslog.
        /// </summary>
        /// <returns>The write.</returns>
        /// <param name="message">The message.</param>
        /// <param name="level">The level. DEBUG, INFO, WARNING, ERROR</param>
        /// <param name="application">The application where the log originates from</param>
        public static void Write(String message, String level, String application = "unknown")
        {
            PostToSyslog(message, level, application).Wait();
        }

        /// <summary>
        /// Grabs a timestamp in syslog format
        /// </summary>
        /// <returns>A timestamp in UTC to send to syslog, e.g. 2018-02-28T21:16:15Z</returns>
        private static String SyslogTimestamp()
        {
            return DateTime.Now.ToUniversalTime().ToString("yyyy-MM-ddTH:mm:ssZ");
        }

        /// <summary>
        /// Builds a JSON Body to send to Syslog
        /// </summary>
        /// <returns>The json body.</returns>
        /// <param name="message">The message.</param>
        /// <param name="level">The level. DEBUG, INFO, WARNING, ERROR</param>
        /// <param name="application">The application where the log originates from</param>
        private static String BuildJsonBody(string message, string level, string application = "unknown")
        {
            var postBody = new Dictionary<string, string>
                {
                    { "message", message },
                    { "service", "Grid" },
                    { "timestamp", SyslogTimestamp() },
                    { "level", level },
                    { "InstanceID", Aws.GetInstanceID()},
                    { "application", application }
                };

            return JsonConvert.SerializeObject(postBody);
        }

        /// <summary>
        /// Posts to syslog.
        /// </summary>
        /// <returns>The to syslog.</returns>
        /// <param name="message">The message.</param>
        /// <param name="level">The level. DEBUG, INFO, WARNING, ERROR</param>
        /// <param name="application">The application where the log originates from</param>
        private static async Task PostToSyslog(string message, string level, string application = "unknown")
        {
            String jsonBody = BuildJsonBody(message, level, application);
            await RequestHelpers.SendPostRequest(endpoint, RequestHelpers.JsonToHttp(jsonBody));
        }
    }
}
