﻿using System;
using System.Diagnostics;
using System.Runtime.InteropServices;
using System.Security;
using System.IO;
using System.Threading;
using Newtonsoft.Json;
using System.Collections.Generic;

namespace AppLauncher
{
    class MainClass
    {
        #region Win32 API Imports

        [DllImport("kernel32.dll")]
        static extern uint WTSGetActiveConsoleSessionId();

        // https://msdn.microsoft.com/en-us/library/windows/desktop/ms684320(v=vs.85).aspx
        [DllImport("kernel32.dll")]
        static extern IntPtr OpenProcess(uint dwDesiredAccess, bool bInheritHandle, uint dwProcessId);

        // https://msdn.microsoft.com/en-us/library/windows/desktop/aa379295(v=vs.85).aspx
        [DllImport("advapi32", SetLastError = true), SuppressUnmanagedCodeSecurityAttribute]
        static extern bool OpenProcessToken(IntPtr ProcessHandle, int DesiredAccess, ref IntPtr TokenHandle);

        [DllImport("kernel32.dll", SetLastError = true)]
        private static extern bool CloseHandle(IntPtr hSnapshot);

        // https://msdn.microsoft.com/en-us/library/windows/desktop/aa446617(v=vs.85).aspx
        [DllImport("advapi32.dll", EntryPoint = "DuplicateTokenEx")]
        public extern static bool DuplicateTokenEx(IntPtr ExistingTokenHandle, uint dwDesiredAccess,
            ref SECURITY_ATTRIBUTES lpThreadAttributes, int TokenType,
            int ImpersonationLevel, ref IntPtr DuplicateTokenHandle);

        // https://msdn.microsoft.com/en-us/library/windows/desktop/ms682429(v=vs.85).aspx
        [DllImport("advapi32.dll", EntryPoint = "CreateProcessAsUser", SetLastError = true, CharSet = CharSet.Ansi, CallingConvention = CallingConvention.StdCall)]
        public extern static bool CreateProcessAsUser(IntPtr hToken, String lpApplicationName, String lpCommandLine, ref SECURITY_ATTRIBUTES lpProcessAttributes,
            ref SECURITY_ATTRIBUTES lpThreadAttributes, bool bInheritHandle, int dwCreationFlags, IntPtr lpEnvironment,
            String lpCurrentDirectory, ref STARTUPINFO lpStartupInfo, out PROCESS_INFORMATION lpProcessInformation);

        #endregion

        #region Structures

        [StructLayout(LayoutKind.Sequential)]
        public struct PROCESS_INFORMATION
        {
            public IntPtr hProcess;
            public IntPtr hThread;
            public uint dwProcessId;
            public uint dwThreadId;
        }

        [StructLayout(LayoutKind.Sequential)]
        public struct SECURITY_ATTRIBUTES
        {
            public int Length;
            public IntPtr lpSecurityDescriptor;
            public bool bInheritHandle;
        }

        // https://msdn.microsoft.com/en-us/library/windows/desktop/ms686331(v=vs.85).aspx
        [StructLayout(LayoutKind.Sequential)]
        public struct STARTUPINFO
        {
            public int cb;
            public String lpReserved;
            public String lpDesktop;
            public String lpTitle;
            public uint dwX;
            public uint dwY;
            public uint dwXSize;
            public uint dwYSize;
            public uint dwXCountChars;
            public uint dwYCountChars;
            public uint dwFillAttribute;
            public uint dwFlags;
            public short wShowWindow;
            public short cbReserved2;
            public IntPtr lpReserved2;
            public IntPtr hStdInput;
            public IntPtr hStdOutput;
            public IntPtr hStdError;
        }

        #endregion

        #region Enumerations

        enum TOKEN_TYPE : int
        {
            TokenPrimary = 1,
            TokenImpersonation = 2
        }

        enum SECURITY_IMPERSONATION_LEVEL : int
        {
            SecurityAnonymous = 0,
            SecurityIdentification = 1,
            SecurityImpersonation = 2,
            SecurityDelegation = 3,
        }

        #endregion

        #region Constants

        public const int TOKEN_DUPLICATE = 0x0002; // https://msdn.microsoft.com/en-us/library/windows/desktop/aa374905(v=vs.85).aspx
        public const uint MAXIMUM_ALLOWED = 0x2000000;

        // Creation Flags

        //https://msdn.microsoft.com/en-us/library/windows/desktop/ms684863(v=vs.85).aspx
        public const int CREATE_NEW_CONSOLE = 0x00000010;
        public const int IDLE_PRIORITY_CLASS = 0x40;
        public const int NORMAL_PRIORITY_CLASS = 0x20;
        public const int HIGH_PRIORITY_CLASS = 0x80;
        public const int REALTIME_PRIORITY_CLASS = 0x100;

        #endregion

        private static string appLaunchPath; // Stores the app to launch. Class Var for logging.

        public static void Main(string[] args)
        {
            bool launchResult;

            if (args.Length == 0) // Ensure the user provides arguments
            {
                WriteLog("Must provide arguments. None were provided.", "ERROR");
                Environment.Exit((1));
            }

            appLaunchPath = args[0];
            PROCESS_INFORMATION procInfo = new PROCESS_INFORMATION();

            WriteLog(("Preparing to launch against path: " + appLaunchPath), "INFO");
            launchResult = Launch(appLaunchPath, out procInfo);

            // If launch result was not true...
            if(!launchResult) {
                WriteLog(("There was a problem launching " + appLaunchPath + ". Please review error logs above"), "WARN");
            }
        }

        public static bool Launch(string applicationName, out PROCESS_INFORMATION procInfo)
        {
            uint explorerPid = 0;
            IntPtr hUserTokenDup = IntPtr.Zero, hPToken = IntPtr.Zero, hProcess = IntPtr.Zero;
            procInfo = new PROCESS_INFORMATION();

            // Get the system session ID
            uint dwSessionId = WTSGetActiveConsoleSessionId();
            WriteLog(("The active session ID located: " + dwSessionId.ToString()), "DEBUG");

            // explorer will indicate the current logged in user
            Process[] processes = Process.GetProcessesByName("explorer");

            // Loop through all of the processes and grab the one of the active user
            foreach (Process p in processes)
            {
                WriteLog(("Checking if process sid " + p.SessionId.ToString() + " matches the active SID " + dwSessionId.ToString()), "DEBUG");
                if ((uint)p.SessionId == dwSessionId) // If the Process Session ID matches the active user session, use that
                {
                    explorerPid = (uint)p.Id;
                    WriteLog(("Found matching PID: " + explorerPid.ToString()), "DEBUG");
                }
            }

            // Opens the existing local process object for explorer, returns an open handle
            hProcess = OpenProcess(MAXIMUM_ALLOWED, false, explorerPid);

            if ( hProcess == null ) {
                int lastErrCode = Marshal.GetLastWin32Error();
                WriteLog("There was a problem opening the process. Last Error: " + lastErrCode.ToString(), "ERROR");
                return false;
            } else {
                WriteLog("The opened process id: " + hProcess.ToString(), "DEBUG");
            }

            // obtain a handle to the access token of the process
            if (!OpenProcessToken(hProcess, TOKEN_DUPLICATE, ref hPToken))
            {
                int lastErrCode = Marshal.GetLastWin32Error();
                WriteLog(("Unable to obtain a process handle. Last Windows Error: " + lastErrCode.ToString()), "ERROR");
                CloseHandle(hProcess); // process token couldn't be returned, close the handle and exit
                return false;
            }

            // Security attibute structure used in DuplicateTokenEx and CreateProcessAsUser
            // I would prefer to not have to use a security attribute variable and to just
            // simply pass null and inherit (by default) the security attributes
            // of the existing token. However, in C# structures are value types and therefore
            // cannot be assigned the null value.
            SECURITY_ATTRIBUTES sa = new SECURITY_ATTRIBUTES();
            sa.Length = Marshal.SizeOf(sa);

            // duplicate the access token of the winlogon process; the newly created token will be a primary token
            if (!DuplicateTokenEx(hPToken, MAXIMUM_ALLOWED, ref sa, (int)SECURITY_IMPERSONATION_LEVEL.SecurityIdentification, (int)TOKEN_TYPE.TokenPrimary, ref hUserTokenDup))
            {
                int lastErrCode = Marshal.GetLastWin32Error();
                WriteLog(("Unable to duplicate the access token. Last Windows Error: " + lastErrCode.ToString()), "ERROR");
                CloseHandle(hProcess);
                CloseHandle(hPToken);
                return false;
            }

            // By default CreateProcessAsUser creates a process on a non-interactive window station, meaning
            // the window station has a desktop that is invisible and the process is incapable of receiving
            // user input. To remedy this we set the lpDesktop parameter to indicate we want to enable user
            // interaction with the new process.
            STARTUPINFO si = new STARTUPINFO();
            si.cb = (int)Marshal.SizeOf(si);
            si.lpDesktop = @"winsta0\default"; // interactive window station parameter; basically this indicates that the process created can display a GUI on the desktop

            // flags that specify the priority and creation method of the process
            int dwCreationFlags = NORMAL_PRIORITY_CLASS | CREATE_NEW_CONSOLE;

            // create a new process in the current user's logon session
            bool result = CreateProcessAsUser(hUserTokenDup,        // client's access token
                                            null,                   // file to execute
                                            applicationName,        // command line
                                            ref sa,                 // pointer to process SECURITY_ATTRIBUTES
                                            ref sa,                 // pointer to thread SECURITY_ATTRIBUTES
                                            false,                  // handles are not inheritable
                                            dwCreationFlags,        // creation flags
                                            IntPtr.Zero,            // pointer to new environment block
                                            null,            // name of current directory
                                            ref si,                 // pointer to STARTUPINFO structure
                                            out procInfo            // receives information about new process
                                            );
            
            WriteLog(("Process started result: " + result.ToString()), "INFO");
            // invalidate the handles
            CloseHandle(hProcess);
            CloseHandle(hPToken);
            CloseHandle(hUserTokenDup);


            return result; // return the result
        }

        // Writes to the logger, prepending the supplied app path
        private static void WriteLog(string message, string level)
        {
            var log = new Dictionary<string, Object>();
            DateTime timestamp = DateTime.Now.ToUniversalTime();

            log.Add("@timestamp", timestamp);
            log.Add("level", level);
            log.Add("message", message);
            log.Add("appLaunchPath", appLaunchPath);

            string jsonLog = JsonConvert.SerializeObject(log);
            Console.WriteLine(jsonLog);
        }
    }
}
