package task /* #nosec */

import (
	"sync"

	"code.justin.tv/qe/automation-webhooks/object"
	"code.justin.tv/qe/automation-webhooks/util"

	"bytes"
	"fmt"
	"log"
	"net/url"
	"os"
	"regexp"
	"strconv"
	"strings"
	"text/template"
	"time"

	"github.com/andygrunwald/go-jira"
	"github.com/google/go-github/github"
	"github.com/pkg/errors"
)

/* #nosec */
const (
	ChangeRequestInfoTitle = "# Change Request Info"
	ChangeRisk             = "Change Risk"
	StartDatetime          = "Change Start Date/Time"
	EndDatetime            = "Change End Date/Time"
	ChangeType             = "Change Type"
	CustomBaseRefToken     = "create cr vs "
	JiraKeyOverride        = "Jira Project Key"
	maxRetries             = 3
)

type CRTemplateData struct {
	FullRepoName string
	Description  string
	LineItems    []string
	PullReqNums  []int
	TriggerURL   string
	TriggerName  string
}

var (
	RiskStrToInt      = map[string]int{"Low": 1, "Medium": 2, "High": 3}
	RiskIntToStr      = map[int]string{1: "Low", 2: "Medium", 3: "High"}
	AutocreationLabel = "Github-CR" // this label indicates a CR ticket is created by automation
	CrBodyTemplate    = `
Auto Generated from Github PR [{{.TriggerName}}|{{.TriggerURL}}]

{{.Description}}

Goals of Change

{{range $i, $e := .LineItems}}  * [{{$e}}|https://git.xarth.tv/{{$.FullRepoName}}/pull/{{index $.PullReqNums 0}}]
{{end}}
`
	ReleaseCrBodyTemplate = `
Auto Generated from Github Release [{{.TriggerName}}|{{.TriggerURL}}]

{{.Description}}

Goals of Change

{{range $i, $e := .LineItems}}  * [{{$e}}|https://git.xarth.tv/{{$.FullRepoName}}/pull/{{index $.PullReqNums $i}}]
{{end}}
`
	PrTemplate = `
Please update the fields below to trigger a Change Request ticket:

# Change Request Info 
## Mandatory Fields 
Change Start Date/Time: YYYY-MM-DD HH:mm 
Change End Date/Time: YYYY-MM-DD HH:mm 
Change Type: Product Launch/Feature Release/Bug Fix/Configuration Change/Datacenter/Network/Maintenance/Experiments/Other 
Change Risk: Low/Medium/High 
Success Criteria:   
 
## Optional Fields 
Self Approve?: Yes/No
:information_source: A field to be used by the creator of the request to self-approve
Approver: 
:information_source: Note: Do not add an approval name if the CR is flagged for self-approval. JIRA will automatically generate an approval request to the individual
Services Impacted: 
:information_source: A list of external services that are part of the [Service Catalog](https://status.internal.justin.tv/services/) who may need to assess the request.
If your change is global, enter "Global Change"
Deployment Plan:  
Rollback Plan:  
Evidence of Testing:    
Architectural Specification: 
Product Specification:  

See documentation at https://wiki.twitch.com/display/QE/User+Guide+to+Automatic+creation+of+Change+Requests
`
)

//nolint
const (
	servicesImpactedField = "Services Impacted"
	selfApproveField      = "Self Approve?"
	approverField         = "Approver"
)

type ChangeRequestInterface interface {
	// each concrete type provides specific implementation.
	ActOnTrigger() (*jira.Issue, error)
	BuildCRDescription(repoName string, newRef string, diff *github.CommitsComparison) (string, error)
}

type ChangeRequestTask struct {
	// common fields
	Github *util.GithubHelper

	// add common (type independent) methods into this struct
}

type PRTriggeredChangeRequestTask struct {
	// fields related to PR trigger
	*ChangeRequestTask
	Comment *object.Comment
	PR      *github.PullRequest
}

type ReleaseTriggeredChangeRequestTask struct {
	// fields related to release trigger
	*ChangeRequestTask
	Release           *object.GithubRelease
	RepositoryRelease *github.RepositoryRelease
}

func max(a, b int) int {
	if a < b {
		return b
	}
	return a
}

func NewPRTriggeredChangeRequestTask(comment *object.Comment) (*PRTriggeredChangeRequestTask, error) {
	githubHelper := &util.GithubHelper{}
	githubHelper.Init()
	pr, _, err := githubHelper.GetPR(comment.Repository.FullName, comment.PullRequest.Number)
	if err != nil {
		return nil, err
	}
	task := &PRTriggeredChangeRequestTask{
		ChangeRequestTask: &ChangeRequestTask{Github: githubHelper},
		Comment:           comment,
		PR:                pr,
	}
	return task, nil
}

func NewReleaseTriggeredChangeRequestTask(releaseEvent *object.GithubRelease) (*ReleaseTriggeredChangeRequestTask, error) {
	githubHelper := &util.GithubHelper{}
	githubHelper.Init()
	release, _, err := githubHelper.GetRelease(releaseEvent.Repository.FullName, releaseEvent.Release.ID)
	if err != nil {
		return nil, err
	}
	task := &ReleaseTriggeredChangeRequestTask{
		ChangeRequestTask: &ChangeRequestTask{Github: githubHelper},
		Release:           releaseEvent,
		RepositoryRelease: release,
	}
	return task, nil
}

func (c *ChangeRequestTask) extractCRFieldsFromText(text string) map[string]string {
	defer logFunctionDuration(time.Now(), "extractCRFieldsFromText")
	customFields := make(map[string]string)
	for _, line := range strings.Split(text, "\n") {
		line = strings.Trim(line, "\r")
		colonPosition := strings.Index(line, ":")
		if colonPosition > 0 {
			foundFieldKey := strings.TrimSpace(line[0:colonPosition])
			foundFieldValue := strings.TrimSpace(line[colonPosition+1:])
			customFields[foundFieldKey] = foundFieldValue
		}
	}
	return customFields
}

//nolint: funlen, lll, unparam
func (c *ChangeRequestTask) computeCRInfo(crBody map[string]string, baseRef string, newRef string, triggerPerson string,
	triggerBody string, triggerTitle string, repoName string, triggerPrNum int, prComments []*github.IssueComment) (*string, map[string]string, []string, *github.CommitsComparison, error) {
	defer logFunctionDuration(time.Now(), "computeCRInfo")

	if trimmedTriggerKeyword := strings.TrimSpace(triggerBody); strings.HasPrefix(strings.ToLower(trimmedTriggerKeyword), CustomBaseRefToken) {
		// user requested to override the base ref (eg video team)
		baseRef = trimmedTriggerKeyword[len(CustomBaseRefToken):]
	}

	jiraProjectKeyCount := make(map[string]int) // keep a counter of potential JIRA project keys
	var jiraIssueKeys []string                  // keep a slice of jira keys

	// Get the diffs between 2 refs (branch names)
	diff, _, err := c.Github.CompareBranches(repoName, baseRef, newRef)
	if err != nil {
		return nil, nil, nil, nil, util.LogAndReturnError(err, "Unable to compare the two branches")
	} else {
		log.Printf("Found %d commits between %s and %s", *diff.TotalCommits, baseRef, newRef)
	}

	// jira ticket search area 1/5 - PR comments
	for _, issueComment := range prComments {
		jiraIssueKeys, jiraProjectKeyCount = c.FindJiraTickets(*issueComment.Body, jiraIssueKeys, jiraProjectKeyCount)
	}

	// jira ticket search area 2/5 - commit message
	// loop through commits in branch, and collect referenced jira tickets
	prBodyChannel := make(chan string)
	var wg sync.WaitGroup
	for _, commit := range diff.Commits {
		jiraIssueKeys, jiraProjectKeyCount = c.FindJiraTickets(*commit.Commit.Message, jiraIssueKeys, jiraProjectKeyCount) // find jira tickets in commit message
		prNumber := c.FindPRNumber(*commit.Commit.Message)
		if prNumber > 0 {
			// there's a reference to another PR - try getting the jira ticket from there
			wg.Add(1)
			go func(result chan<- string, wg *sync.WaitGroup) { // get PRs concurrently
				defer wg.Done()
				var linkedPR *github.PullRequest
				for i := 1; i <= maxRetries; i++ { // backoff-retry github request
					linkedPR, _, err = c.Github.GetPR(repoName, prNumber)
					if err != nil {
						log.Printf("Unable to get %s PR %d due to %s", repoName, prNumber, err.Error())
						if i < maxRetries {
							log.Printf("will backoff and retry")
							time.Sleep(time.Duration(i) * 100 * time.Millisecond)
						}
						continue
					}
					break
				}
				if *linkedPR.Body != "" {
					result <- *linkedPR.Body
				}
			}(prBodyChannel, &wg)
		}
	}
	go func() { // wait for concurrent PR requests to complete
		wg.Wait()
		close(prBodyChannel)
	}()

	for prBody := range prBodyChannel {
		jiraIssueKeys, jiraProjectKeyCount = c.FindJiraTickets(prBody, jiraIssueKeys, jiraProjectKeyCount)
	}

	// jira ticket search area 3/5 - branch name
	jiraIssueKeys, jiraProjectKeyCount = c.FindJiraTickets(newRef, jiraIssueKeys, jiraProjectKeyCount)
	// jira ticket search area 4/5 - PR body (the original comment)
	jiraIssueKeys, jiraProjectKeyCount = c.FindJiraTickets(triggerBody, jiraIssueKeys, jiraProjectKeyCount)
	// jira ticket search area 5/5 - PR title
	jiraIssueKeys, jiraProjectKeyCount = c.FindJiraTickets(triggerTitle, jiraIssueKeys, jiraProjectKeyCount)

	// Find the best match jira project key (most frequently referenced)
	jiraProjectKey := ""
	if val, ok := crBody[JiraKeyOverride]; ok {
		log.Printf("User overriding %s = %s", JiraKeyOverride, val)
		jiraProjectKey = val
	} else {
		jiraProjectKey = c.findMostFrequentlyReferencedJiraProjectKey(jiraProjectKeyCount)
	}
	if jiraProjectKey == "" {
		// can't create CR as we don't know the jira project key
		return nil, nil, nil, nil, util.LogAndReturnError(nil, "Unable to find any JIRA project key in commit messages and branch name")
	}
	log.Printf("Found most-frequently-referenced JIRA project key %s", jiraProjectKey)

	// get CR fields
	customFields := crBody
	jiraUser, err := c.FindJiraUser(triggerPerson)
	if err != nil {
		// do not exit. proceed to create CR ticket with the default user for the project
		log.Printf("Warning: Can't override Reporter field due to failure looking up github user %s's JIRA user", triggerPerson)
	} else {
		customFields["Reporter"] = jiraUser // override default reporter - requested by QE-1476
	}
	return &jiraProjectKey, customFields, jiraIssueKeys, diff, nil
}

// doPreCreation sets up two maps containing jira fields -- one for the subsequent jira ticket creation, and the other
// for the next step of updating the newly jira ticket.
// Currently it populates these fields: Services Impacted, Self-Approve, Approver
func (c *ChangeRequestTask) doPreCreation(fullRepoName string, creationFields map[string]string) (map[string]string, map[string]string) {
	defer logFunctionDuration(time.Now(), "doPreCreation")
	updateFields := make(map[string]string)
	dynamodb := util.NewDynamoDBHelper("repodata-" + os.Getenv("ENVIRONMENT")) // beanstalk devtooling guarantees this env var exists
	if val, ok := creationFields[servicesImpactedField]; ok && len(val) > 0 {
		updateFields[servicesImpactedField] = val
		delete(creationFields, servicesImpactedField)
	} else {
		persistedRepoData, err := dynamodb.GetRepoData(fullRepoName)
		if err != nil {
			log.Printf("Error reading from dynamodb: %s", err)
		} else if len(persistedRepoData.ServicesImpacted) > 0 {
			updateFields[servicesImpactedField] = strings.Join(persistedRepoData.ServicesImpacted, ",")
		}
	}
	return creationFields, updateFields
}

// createCRTicket creates CR ticket in 2 stages:
// stage 1: create the ticket
// stage 2: update the ticket
// This is to increase the chances of getting a CR created once there is sufficient data. The update step will add non-critical data.
func (c *ChangeRequestTask) createCRTicket(jiraProjectKey string, triggerTitle string, crDescription string,
	creationFields map[string]string, updateFields map[string]string, jiraIssueKeys []string) (*jira.Issue, error) {
	// create CR in jira
	jiraClient := util.GetJiraHelperInstance()
	persistedIssue, err := jiraClient.CreateIssue(jiraProjectKey, "Change Request", triggerTitle,
		crDescription, creationFields)
	if err != nil {
		return nil, err
	} else {
		log.Printf("Created CR ticket %s", persistedIssue.Key)
		// set label to keep track of auto-generated tickets - QE-1475
		persistedIssue, err := jiraClient.GetIssue(persistedIssue.Key)
		if err != nil {
			return nil, util.LogAndReturnError(err, "Failed to retrieve persisted CR ticket")
		}
		// link all the related jira tickets to the CR - QE-1526
		persistedIssue, err = jiraClient.LinkIssues(persistedIssue.Key, jiraIssueKeys, "Relates")
		if err != nil {
			return nil, err
		}
		allTheLabels := append(persistedIssue.Fields.Labels, AutocreationLabel)
		updateFields["Labels"] = strings.Join(allTheLabels, ",")
		persistedIssue, err = jiraClient.UpdateIssue(persistedIssue.Key, updateFields)
		return persistedIssue, err
	}
}

// doPostCreation performs post-processing tasks (eg persists repo related information to dynamo)
func (c *ChangeRequestTask) doPostCreation(fullRepoName string, cr *jira.Issue) {
	defer logFunctionDuration(time.Now(), "doPostCreation")
	if fullRepoName == "" || cr == nil {
		log.Printf("Invalid/Missing inputs to doPostCreation(): fullRepoName and/or jira issue. Returning without further processing.")
		return
	}
	dynamodb := util.NewDynamoDBHelper("repodata-" + os.Getenv("ENVIRONMENT")) // beanstalk devtooling guarantees this env var exists
	_, dynErr := dynamodb.PutRepoData(fullRepoName, cr)
	if dynErr != nil {
		log.Printf("Failed to persist repo data due to: %s", dynErr)
	}
}

// FindJiraTickets finds jira tickets in specified text, and updates the map containing jira ticket and respective counts
func (c *ChangeRequestTask) FindJiraTickets(text string, jiraIssueKeys []string, jiraProjectKeyCount map[string]int) (keys []string, counts map[string]int) {
	defer logFunctionDuration(time.Now(), "FindJiraTickets")
	// Find potential jira project keys and count them. When processing potentially large number of messages,
	// there could be cases of cross-referencing jira tickets, eg "QE-123 fixes a dependent issue in DTA-456".
	// In general, the most-frequently repeated project is is the one we're looking for.
	jiraProjectKeyRegex := regexp.MustCompile(`([A-Za-z]{2,}-[0-9]{1,})`)
	keyMatches := jiraProjectKeyRegex.FindStringSubmatch(text)
	if len(keyMatches) > 0 {
		for _, match := range keyMatches {
			jiraIssueKeys = append(jiraIssueKeys, match)
			projectKey := strings.ToUpper(strings.Split(match, "-")[0])
			log.Printf("Found a jira project reference in commit message: %s", projectKey)
			val, ok := jiraProjectKeyCount[projectKey]
			if ok {
				jiraProjectKeyCount[projectKey] = val + 1
			} else {
				jiraProjectKeyCount[projectKey] = 1
			}
		}
	}
	return jiraIssueKeys, jiraProjectKeyCount
}

// FindPRNumber finds a PR number in specified text. Returns -1 if not found
func (c *ChangeRequestTask) FindPRNumber(text string) int {
	prNumRegex := regexp.MustCompile(`(Merge pull request #)([0-9]+)( from.*)`)
	prMatches := prNumRegex.FindStringSubmatch(text) // find pr # in commit message
	if len(prMatches) == 0 {
		return -1
	}
	prNumber, err := strconv.Atoi(prMatches[2]) // group #3 contains PR number
	if err != nil {
		log.Printf("Unable to parse PR number")
		return -1
	}
	return prNumber
}

// ResolveFields compares the new map of custom fields and the previously resolved map, and returns a new map that
// contains values that are determined to best describe the CR (in terms of data quality)
func (c *ChangeRequestTask) ResolveFields(resolvedMap map[string]string, newMap map[string]string) map[string]string {
	// Lets do a bit better than keeping the latest values of the custom fields:
	// 1) Change Risk = Max(all change risks evaluated)
	// 2) Evidence fields = Concatenate unique values
	// Also allow ease of customization in the future.
	for key := range resolvedMap { // for each key in overall field set
		currentValue, ok := newMap[key]
		if ok { // if key exists in new
			newValue := currentValue
			// compare overall and new value
			switch key {
			case ChangeRisk:
				// overall change risk = max of individual change risks
				currentRisk := RiskStrToInt[currentValue]
				overallRisk := RiskStrToInt[resolvedMap[ChangeRisk]]
				newValue = RiskIntToStr[(max(currentRisk, overallRisk))]
			case StartDatetime:
				// nothing special
			case EndDatetime:
				// nothing special
			case ChangeType:
				// nothing special
			default:
				// append new value if not already done so
				prevValue := resolvedMap[key]
				if !strings.Contains(prevValue, newValue) {
					newValue = resolvedMap[key] + ". " + newValue
				}
			}
			// set evaluated overall value, as of this iteration
			resolvedMap[key] = newValue
		} // else overall map is unchanged
	}
	for key := range newMap {
		_, ok := resolvedMap[key]
		if !ok {
			resolvedMap[key] = newMap[key]
		}
	}
	for key := range resolvedMap {
		log.Printf("Overall custom field key=%s, value=%s", key, resolvedMap[key])
	}
	return resolvedMap
}

// findMostFrequentlyReferencedJiraProjectKey returns the most frequently referenced jira project key
func (c *ChangeRequestTask) findMostFrequentlyReferencedJiraProjectKey(jiraProjectKeyCount map[string]int) string {
	jiraProjectKey := ""
	maxJiraKeyCount := -1
	for key := range jiraProjectKeyCount {
		val := jiraProjectKeyCount[key]
		if val > maxJiraKeyCount {
			jiraProjectKey = key
			maxJiraKeyCount = val
		}
	}
	return jiraProjectKey
}

// FindJiraUser tries to find a jira user that maps to the specified github user. This needs to be done due to
// some inconsistent cases where github.userid != jira.userid
func (c *ChangeRequestTask) FindJiraUser(githubLogin string) (string, error) {
	defer logFunctionDuration(time.Now(), "FindJiraUser")
	githubUser, _, err := c.Github.GetUser(githubLogin)
	if err != nil {
		return "", err
	}
	if githubUser == nil {
		return "", errors.New(fmt.Sprintf("Could not find github user with id %s", githubLogin))
	}

	// search strings
	searchStrings := []string{
		githubLogin + "@twitch.tv", // attempt to find by login@twitch.tv
		githubLogin + "@justin.tv", // attempt to find by login@justin.tv
		githubLogin,                // attempt to find by login
	}

	if githubUser.Email != nil {
		// pre-pend with configured email address (if it exists...)
		searchStrings = append([]string{*githubUser.Email}, searchStrings...)
	}

	// last resort - use full name
	if len(*githubUser.Name) > 0 {
		searchStrings = append(searchStrings, url.QueryEscape(*githubUser.Name))
	}

	for _, searchString := range searchStrings {
		users, _, err := util.GetJiraHelperInstance().FindUsers(searchString) // there should be 0 or 1 matches
		if err != nil {
			continue
		}
		if len(*users) == 0 {
			continue
		}
		return (*users)[0].Name, nil
	}

	return "", errors.New("Could not find any user")
}

func (c *PRTriggeredChangeRequestTask) BuildCRDescription(repoName string, newRef string,
	diff *github.CommitsComparison) (string, error) {
	var lineItems []string
	var prNums []int
	sanitizer := strings.NewReplacer("\n\n", ". ", "[", "\\[", "]", "\\]", "{", "\\{", "}", "\\}") // escape special characters that impact the templates
	var templ *template.Template
	triggerURL := fmt.Sprintf("https://git.xarth.tv/%s/pull/%d", repoName, *c.PR.Number)
	triggerLinkName := fmt.Sprintf("%s #%d", repoName, *c.PR.Number)

	if strings.HasPrefix(strings.ToLower(newRef), "release/") {
		// github "release" - collect PR merge commit messages (generated by github) only
		for _, commit := range diff.Commits {
			prNumber := c.FindPRNumber(*commit.Commit.Message)
			if prNumber > 0 {
				clean := sanitizer.Replace(strings.TrimSpace(*commit.Commit.Message))
				prNums = append(prNums, prNumber)
				lineItems = append(lineItems, clean)
			}
		}
		templ = template.Must(template.New("template").Parse(CrBodyTemplate))
	} else {
		// regular feature branch PR - collect commit messages
		prNums = append(prNums, *c.PR.Number)
		for _, commit := range diff.Commits {
			clean := sanitizer.Replace(strings.TrimSpace(*commit.Commit.Message))
			lineItems = append(lineItems, clean)
		}
		templ = template.Must(template.New("template").Parse(CrBodyTemplate))
	}
	crTemplateData := &CRTemplateData{
		TriggerURL:   triggerURL,
		TriggerName:  triggerLinkName,
		Description:  strings.TrimSpace(strings.Split(*c.PR.Body, ChangeRequestInfoTitle)[0]),
		FullRepoName: repoName,
		LineItems:    lineItems,
		PullReqNums:  prNums,
	}
	crDescription := bytes.NewBufferString("")
	err := templ.Execute(crDescription, crTemplateData)
	return crDescription.String(), err
}

// ActOnPRComment creates CR as requested by a PR comment. To complete successfully, the parent PR requires:
// 1) the associated github branch name to contain a jira ticket (eg to a story, bug, etc)
// 2) the PR description contains the token CHANGE_REQUEST_INFO_TITLE, followed by field name-value pairs, eg:
//		# Change Request Info
//		Change Start Date/Time:	YYYY-MM-DD HH:mm
//		Change End Date/Time:	YYYY-MM-DD HH:mm
//		Change Type: Feature Release/Bugfix/Configuration change
//		Change Risk: Low/Medium/High
//		Success Criteria:
//		Evidence of Testing:
//		Architectural Specification:
//		Product Specification:
//		Deployment Plan:
//		Rollback Plan:
func (c *PRTriggeredChangeRequestTask) ActOnTrigger() (*jira.Issue, error) {
	defer logFunctionDuration(time.Now(), "ActOnTrigger")
	// get PR
	comment := *c.Comment
	pr, _, err := c.Github.GetPR(comment.Repository.FullName, comment.PullRequest.Number)
	if err != nil {
		return nil, util.LogAndReturnError(err, "Error retrieving PR. Not proceeding with CR creation")
	}
	log.Printf("Found PR #%d in repo %s", *pr.Number, *pr.Head.Repo.FullName)

	crFields := c.getCRFieldsFromPR(&comment, pr)
	if crFields == nil {
		return nil, c.addCRTemplateToPR(&comment)
	}
	prComments, _, err := c.Github.ListPRComments(comment.Repository.FullName, comment.PullRequest.Number)
	if err != nil {
		log.Printf("Unable to retrieve comments from PR #%d in repo %s. This is not necessarily a problem",
			comment.PullRequest.Number, comment.Repository.FullName) // find jira ticket reference in PR comment
	}
	jiraProjectKey, customFields, jiraIssueKeys, diff, err := c.computeCRInfo(crFields, *pr.Base.Ref, *pr.Head.Ref,
		comment.Sender.Login, comment.Comment.Body, *pr.Title, comment.Repository.FullName, *pr.Number, prComments)
	if err != nil {
		return nil, err
	}
	crDescription, err := c.BuildCRDescription(comment.Repository.FullName, *pr.Head.Ref, diff)
	if err != nil {
		return nil, err
	}
	creationFields, updateFields := c.doPreCreation(comment.Repository.FullName, customFields)
	cr, err := c.createCRTicket(*jiraProjectKey, *pr.Title, crDescription, creationFields, updateFields, jiraIssueKeys)
	c.doPostCreation(comment.Repository.FullName, cr)
	return cr, err
}

func (c *PRTriggeredChangeRequestTask) getCRFieldsFromPR(comment *object.Comment, pr *github.PullRequest) map[string]string {
	if pr != nil && strings.Contains(*pr.Body, ChangeRequestInfoTitle) {
		return c.extractCRFieldsFromText(*pr.Body)
	} else if strings.Contains(comment.Comment.Body, ChangeRequestInfoTitle) {
		return c.extractCRFieldsFromText(comment.Comment.Body)
	}
	return nil
}

func (c *PRTriggeredChangeRequestTask) addCRTemplateToPR(comment *object.Comment) error {
	_, _, err := c.Github.CreatePRReviewComment(comment.Repository.FullName, comment.PullRequest.Number, PrTemplate)
	if err != nil {
		log.Printf("CR data section not found, but unable to create comment with PR template due to %s", err.Error()) // this shouldn't happen. log it anyway.
		return err
	}
	log.Printf("CR data section not found. Created comment containing PR template")
	return nil
}

func (c *ReleaseTriggeredChangeRequestTask) ActOnTrigger() (*jira.Issue, error) {
	defer logFunctionDuration(time.Now(), "ActOnTrigger")
	// get release
	release := c.Release
	crFields := c.extractCRFieldsFromText(release.Release.Body)
	releases, _, err := c.Github.ListReleases(release.Repository.FullName)
	if err != nil {
		log.Printf("Unable to retrieve github releases for %s due to: %s", release.Repository.FullName, err.Error())
		return nil, err
	}
	previousRelease := releases[1]
	jiraProjectKey, customFields, jiraIssueKeys, diff, err := c.computeCRInfo(crFields, *previousRelease.TagName,
		release.Release.Target, release.Sender.Login, release.Release.Body, release.Release.Name,
		release.Repository.FullName, -1, nil)
	if err != nil {
		return nil, err
	}
	crDescription, err := c.BuildCRDescription(release.Repository.FullName, release.Release.Target, diff)
	if err != nil {
		return nil, err
	}
	creationFields, updateFields := c.doPreCreation(release.Repository.FullName, customFields)
	cr, err := c.createCRTicket(*jiraProjectKey, release.Release.Name, crDescription, creationFields, updateFields, jiraIssueKeys)
	c.doPostCreation(release.Repository.FullName, cr)
	return cr, err
}

func (c *ReleaseTriggeredChangeRequestTask) BuildCRDescription(repoName string, newRef string,
	diff *github.CommitsComparison) (string, error) {
	var lineItems []string
	var prNums []int
	sanitizer := strings.NewReplacer("\n\n", ". ", "[", "\\[", "]", "\\]", "{", "\\{", "}", "\\}") // escape special characters that impact the templates
	var templ *template.Template
	triggerURL := fmt.Sprintf("https://git.xarth.tv/%s/releases/tag/%s", repoName, c.Release.Release.TagName)
	triggerLinkName := fmt.Sprintf("%s %s", repoName, c.Release.Release.TagName)
	for _, commit := range diff.Commits {
		prNumber := c.FindPRNumber(*commit.Commit.Message)
		if prNumber > 0 {
			clean := sanitizer.Replace(strings.TrimSpace(*commit.Commit.Message))
			prNums = append(prNums, prNumber)
			lineItems = append(lineItems, clean)
		}
	}
	templ = template.Must(template.New("template").Parse(ReleaseCrBodyTemplate))
	crTemplateData := &CRTemplateData{
		TriggerURL:   triggerURL,
		TriggerName:  triggerLinkName,
		Description:  strings.TrimSpace(strings.Split(c.Release.Release.Body, ChangeRequestInfoTitle)[0]),
		FullRepoName: repoName,
		LineItems:    lineItems,
		PullReqNums:  prNums,
	}
	crDescription := bytes.NewBufferString("")
	err := templ.Execute(crDescription, crTemplateData)
	return crDescription.String(), err
}

func logFunctionDuration(start time.Time, functionName string) {
	elapsed := time.Since(start)
	log.Printf("%s took %s", functionName, elapsed)
}
