package util

import (
	"log"
	"strings"

	"code.justin.tv/qe/automation-webhooks/object"
	"github.com/andygrunwald/go-jira"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
)

const (
	DefaultRegion = "us-west-2"
)

// DynamoDBHelper struct
type DynamoDBHelper struct {
	TableName string
}

// NewDynamoDBHelper instantiates a dynamodb helper
func NewDynamoDBHelper(tableName string) *DynamoDBHelper {
	return &DynamoDBHelper{
		TableName: tableName,
	}
}

// getDynamoDB returns dynamodb service client
func (d *DynamoDBHelper) getDynamoDB() (*dynamodb.DynamoDB, error) {
	sess, err := session.NewSession(&aws.Config{Region: aws.String(DefaultRegion)})
	if err != nil {
		return nil, err
	}

	return dynamodb.New(sess), nil
}

// GetRepoData returns repo data associated with the fullRepoName
func (d *DynamoDBHelper) GetRepoData(fullRepoName string) (*object.RepoItem, error) {
	dynamoService, err := d.getDynamoDB()
	if err != nil {
		return nil, err
	}

	dynamoItem := &dynamodb.GetItemInput{
		TableName: aws.String(d.TableName),
		Key: map[string]*dynamodb.AttributeValue{
			"repo": {
				S: aws.String(fullRepoName),
			},
		},
	}

	result, err := dynamoService.GetItem(dynamoItem)

	if err != nil {
		return nil, err
	}

	itemModel := &object.RepoItem{}
	err = dynamodbattribute.UnmarshalMap(result.Item, itemModel)
	return itemModel, err
}

// PutRepoData persists repo data for the associated fullRepoName
func (d *DynamoDBHelper) PutRepoData(fullRepoName string, issue *jira.Issue) (*object.RepoItem, error) {
	dynamoService, err := d.getDynamoDB()
	if err != nil {
		return nil, err
	}

	repoItemModel := &object.RepoItem{
		Repo:           fullRepoName, // key
		JiraProjectKey: strings.Split(issue.Key, "-")[0],
	}

	servicesImpacted, err := issue.Fields.Unknowns.String(JiraFields["Services Impacted"])
	if err == nil {
		repoItemModel.ServicesImpacted = strings.Split(servicesImpacted, multiselectSeparator)
	}

	attrMap, err := dynamodbattribute.MarshalMap(repoItemModel)
	if err != nil {
		log.Fatalf("failed to marshal due to: %s", err.Error())
	}
	dynamoItem := &dynamodb.PutItemInput{
		TableName: aws.String(d.TableName),
		Item:      attrMap,
	}

	_, err = dynamoService.PutItem(dynamoItem)

	return repoItemModel, err
}
