package util

import (
	"context"
	"log"
	"net/url"
	"os"
	"strings"

	"github.com/google/go-github/github"
	"golang.org/x/oauth2"
)

type GithubHelper struct {
	client *github.Client
}

const (
	GithubBaseURL = "https://git.xarth.tv/api/v3/"
)

func (g *GithubHelper) Init() {
	accessToken := os.Getenv("GITHUB_ACCESS_TOKEN") // devtools' tooling on terraform + beanstalk provides this
	if accessToken == "" {
		env := os.Getenv("ENVIRONMENT") // devtools' tooling on terraform + beanstalk provides this
		accessTokenSS, err := GetSandstormHelperInstance().GetSecrets("qa-eng/oml/" + env + "/ghe")
		if err != nil {
			log.Fatalf("Hard failure: unable to get github access token from environment or sandstorm. Can't instantiate helper.")
		} else {
			accessToken = accessTokenSS
		}
	}

	ts := oauth2.StaticTokenSource(
		&oauth2.Token{AccessToken: accessToken},
	)
	tc := oauth2.NewClient(context.Background(), ts)
	g.client = github.NewClient(tc)
	baseURL, err := url.Parse(GithubBaseURL)
	if err != nil {
		log.Panicf("error parsing github base url")
	}
	g.client.BaseURL = baseURL
}

func (g *GithubHelper) getOwnerRepo(fullRepoName string) (string, string) {
	tokens := strings.Split(fullRepoName, "/")
	return tokens[0], tokens[1]
}

func (g *GithubHelper) GetPR(fullRepoName string, prNumber int) (*github.PullRequest, *github.Response, error) {
	owner, repo := g.getOwnerRepo(fullRepoName)
	return g.client.PullRequests.Get(context.Background(), owner, repo, prNumber)
}

func (g *GithubHelper) ListPRComments(fullRepoName string, prNumber int) ([]*github.IssueComment, *github.Response, error) {
	owner, repo := g.getOwnerRepo(fullRepoName)
	options := &github.IssueListCommentsOptions{
		ListOptions: github.ListOptions{
			PerPage: 1000,
		},
	}
	return g.client.Issues.ListComments(context.Background(), owner, repo, prNumber, options)
}

func (g *GithubHelper) ListPRs() {
}

func (g *GithubHelper) ListReleases(fullRepoName string) ([]*github.RepositoryRelease, *github.Response, error) {
	owner, repo := g.getOwnerRepo(fullRepoName)
	options := &github.ListOptions{
		PerPage: 1000,
	}
	return g.client.Repositories.ListReleases(context.Background(), owner, repo, options)
}

func (g *GithubHelper) CompareBranches(fullRepoName string, baseReference string, newReference string) (*github.CommitsComparison, *github.Response, error) {
	owner, repo := g.getOwnerRepo(fullRepoName)
	return g.client.Repositories.CompareCommits(context.Background(), owner, repo, baseReference, newReference)
}

// Create a new PR Review Comment with the provided text
func (g *GithubHelper) CreatePRReviewComment(fullRepoName string, prNumber int, text string) (*github.IssueComment, *github.Response, error) {
	owner, repo := g.getOwnerRepo(fullRepoName)
	issueComment := &github.IssueComment{
		Body: &text,
	}
	return g.client.Issues.CreateComment(context.Background(), owner, repo, prNumber, issueComment)
}

// Edit a new PR Review Comment with the provided text
func (g *GithubHelper) UpdatePRReviewComment(fullRepoName string, commentID int, text string) (*github.IssueComment, *github.Response, error) {
	owner, repo := g.getOwnerRepo(fullRepoName)
	issueComment := &github.IssueComment{
		Body: &text,
	}
	return g.client.Issues.EditComment(context.Background(), owner, repo, commentID, issueComment)
}

// GetUser returns a User
func (g *GithubHelper) GetUser(userName string) (*github.User, *github.Response, error) {
	return g.client.Users.Get(context.Background(), userName)
}

// Get release
func (g *GithubHelper) GetRelease(fullRepoName string, id int64) (*github.RepositoryRelease, *github.Response, error) {
	owner, repo := g.getOwnerRepo(fullRepoName)
	return g.client.Repositories.GetRelease(context.Background(), owner, repo, id)
}

// Update release body/description
func (g *GithubHelper) UpdateReleaseBody(fullRepoName string, id int64, body string) (*github.RepositoryRelease, *github.Response, error) {
	release, resp, err := g.GetRelease(fullRepoName, id)
	if err != nil {
		return nil, resp, err
	}
	owner, repo := g.getOwnerRepo(fullRepoName)
	release.Body = &body
	return g.client.Repositories.EditRelease(context.Background(), owner, repo, id, release)
}
