package util

import (
	"crypto/rsa"
	"crypto/x509"
	"encoding/pem"
	"io/ioutil"
	"log"
	"os"
	"path/filepath"
	"strings"
	"sync"

	"context"
	"fmt"
	"time"

	"github.com/andygrunwald/go-jira"
	"github.com/dghubble/oauth1"
	"github.com/pkg/errors"
	"github.com/trivago/tgo/tcontainer"
)

// See https://gist.github.com/Lupus/edafe9a7c5c6b13407293d795442fe67

type JiraHelper struct {
	jiraURL    string
	roleArn    string
	jiraClient *jira.Client
}

/* #nosec */
const (
	ssPrefix             = "qa-eng/automation-webhooks/"
	consumerkey          = "consumerkey"
	tokenfile            = "jira_token.auth"
	keyfile              = "jira-privkey.pem"
	multiselectSeparator = "; "
	ReferenceLocale      = "America/Los_Angeles"
)

// Automation-friendly input field names to JIRA-internal field keys
// ie user-input space -> JIRA space
var (
	JiraFields = map[string]string{
		"Change Start Date/Time":      "customfield_20100",
		"Change End Date/Time":        "customfield_20101",
		"Change Type":                 "customfield_20000",
		"Change Risk":                 "customfield_20002",
		"Success Criteria":            "customfield_20003",
		"Evidence of Testing":         "customfield_20103",
		"Architectural Specification": "customfield_20104",
		"Product Specification":       "customfield_20105",
		"Deployment Plan":             "customfield_20106",
		"Rollback Plan":               "customfield_20300",
		"Self Approve?":               "customfield_20007",
		"Reporter":                    "Reporter",
		"Labels":                      "Labels",
		"Platform":                    "Labels",
		"Services Impacted":           "customfield_21321", // dev jira: 21700, prod jira: 21321
		"Approver":                    "customfield_20006",
	}
	ServicesImpacted = map[string]string{ // these fields are temporary out of sync between dev and production jira
		"production": "customfield_21321", // FIXME: remove the workaround after prod and dev are synced
		"staging":    "customfield_21700",
	}
	jiraInstance *JiraHelper
	jiraOnce     sync.Once
)

// GetSandstormHelperInstance returns the singleton instance of SandstormHelper.
// This uses reference implementation from http://marcio.io/2015/07/singleton-pattern-in-go/
func GetJiraHelperInstance() *JiraHelper {
	jiraOnce.Do(func() {
		env := os.Getenv("ENVIRONMENT") // devtools' tooling on terraform + beanstalk provides this
		// Reference: roleArn should be the IAM role that has the sts:assumeRole policy
		// The role should be defined in terraform/twitch-cape-qe-aws/sandstorm_<env>.tf
		// The role should be allowed to access secret pattern: qa-eng/oml/<env>/*
		roleArn := "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/qa-eng-" + env + "-automation-webhooks"

		jiraURL := os.Getenv("JIRA_URL")
		if jiraURL == "" {
			jiraURLSS, err := GetSandstormHelperInstance().GetSecrets(ssPrefix + env + "/jira-url")
			if err != nil {
				log.Fatalf("Hard failure: unable to determine jira url from environment or sandstorm. Can't instantiate helper.")
			} else {
				jiraURL = jiraURLSS
			}
		}
		jiraInstance = &JiraHelper{
			jiraURL: jiraURL,
			roleArn: roleArn,
		}
		jiraInstance.Init()
	})
	return jiraInstance
}

func (j *JiraHelper) GetJiraURL() string {
	return j.jiraURL
}

func (j *JiraHelper) readJiraToken() (string, error) {
	fullPath, err := filepath.Abs(tokenfile)
	if err != nil {
		log.Fatal("unable to construct absolute path to file")
	}
	/* #nosec */
	tokenBytes, err := ioutil.ReadFile(fullPath)
	if err != nil {
		// get from sandstorm
		env := os.Getenv("ENVIRONMENT") // devtools' tooling on terraform + beanstalk provides this
		tokenBytes, err = GetSandstormHelperInstance().GetSecretsRaw(ssPrefix + env + "/" + tokenfile)
	}
	if err != nil {
		log.Fatal("unable to get auth token from filesystem or sandstorm")
		return "", err
	}
	return string(tokenBytes), err
}

func (j *JiraHelper) readJiraPrivateKey() (*rsa.PrivateKey, error) {
	fullPath, err := filepath.Abs(keyfile)
	if err != nil {
		log.Fatal("unable to construct absolute path to file")
	}
	/* #nosec */
	privKeyBytes, err := ioutil.ReadFile(fullPath)
	if err != nil {
		// get from sandstorm
		env := os.Getenv("ENVIRONMENT") // devtools' tooling on terraform + beanstalk provides this
		privKeyBytes, err = GetSandstormHelperInstance().GetSecretsRaw(ssPrefix + env + "/" + keyfile)
		if err != nil {
			log.Panicf("unable to get secrets from sandstorm due to: %s", err.Error())
		}
	}

	keyDERBlock, _ := pem.Decode(privKeyBytes)

	if keyDERBlock == nil {
		log.Fatal("unable to decode key PEM block")
	}
	if !(keyDERBlock.Type == "PRIVATE KEY" || strings.HasSuffix(keyDERBlock.Type, " PRIVATE KEY")) {
		log.Panicf("unexpected key DER block type: %s", keyDERBlock.Type)
	}

	privateKey, err := x509.ParsePKCS1PrivateKey(keyDERBlock.Bytes)
	return privateKey, err
}

func (j *JiraHelper) Init() {
	privateKey, err := j.readJiraPrivateKey()
	if err != nil {
		log.Panicf("unable to parse PKCS1 private key. %v", err)
	}

	consumerKey := os.Getenv(consumerkey)
	if consumerKey == "" {
		env := os.Getenv("ENVIRONMENT") // devtools' tooling on terraform + beanstalk provides this
		consumerKey, err = GetSandstormHelperInstance().GetSecrets(ssPrefix + env + "/" + consumerkey)
		if err != nil {
			log.Panicf("failed to get secrets due to: %s", err.Error())
		}
	}
	oauthConfig := &oauth1.Config{
		ConsumerKey: consumerKey,
		Signer:      &oauth1.RSASigner{PrivateKey: privateKey},
	}

	tokensString, err := j.readJiraToken()
	if err != nil {
		log.Panicf("failed to read jira token due to: %s", err)
	}
	splitTokens := strings.Split(tokensString, ":") // token file contains <key>:<secret>
	httpClient := oauth1.NewClient(context.Background(), oauthConfig, oauth1.NewToken(splitTokens[0], splitTokens[1]))
	newClient, err := jira.NewClient(httpClient, j.jiraURL)
	if err != nil {
		log.Panicf("unable to create new JIRA client. %v", err)
	}
	j.jiraClient = newClient

}
func (j *JiraHelper) SearchIssues(jql string) ([]jira.Issue, *jira.Response, error) {
	return j.jiraClient.Issue.Search(jql, nil)
}

// CreateIssue creates jira tickets. It takes in caller-friendly inputFields and internally converts to
// jira backend friendly data structures. It also understands Twitch's custom fields.
func (j *JiraHelper) CreateIssue(projectKey string, issueType string, summary string, description string, inputFields map[string]string) (*jira.Issue, error) {
	customFieldContainer := tcontainer.NewMarshalMap()

	// set standard fields
	ticket := &jira.Issue{
		Fields: &jira.IssueFields{
			Project:     jira.Project{Key: projectKey},
			Type:        jira.IssueType{Name: issueType},
			Summary:     summary,
			Description: description,
			Unknowns:    customFieldContainer,
		},
	}

	preCheckErr := j.populateFieldsWithStructuredData(ticket, inputFields)
	persistedIssue, resp, err := j.jiraClient.Issue.Create(ticket)

	if err != nil {
		return nil, LogAndReturnError(nil, "Unable to create issue due to: %s", j.GetResponseBodyString(resp))
	}
	if preCheckErr != nil {
		return persistedIssue, preCheckErr
	}

	return persistedIssue, err
}

// GetResponseBodyString returns a string representation of the jira response body
func (j *JiraHelper) GetResponseBodyString(response *jira.Response) string {
	bodyBytes, err := ioutil.ReadAll(response.Body)
	if err != nil {
		return ""
	} else {
		return string(bodyBytes)
	}
}

func (j *JiraHelper) populateFieldsWithStructuredData(ticket *jira.Issue, inputFields map[string]string) error { //nolint: funlen, gocyclo
	var validationError error
	// set jira ticket fields based on Twitch's convention/process
	if ticket.Fields.Unknowns == nil {
		ticket.Fields.Unknowns = tcontainer.NewMarshalMap()
	}
	for key := range inputFields {
		fieldID, ok := JiraFields[key]
		if ok {
			switch key {
			// NOTE: make sure to comment on the process/logic/rule for handling the custom fields.
			// Reference jira ticket(s) if available.
			case "Change Start Date/Time", "Change End Date/Time":
				// customfield type = datetime
				// convert user input YYYY-MM-DD hh:mm to jira-friendly YYYY-MM-DDThh:mm:00+0000
				tokens := strings.Split(inputFields[key], " ")
				if len(tokens) == 2 {
					referenceLocation, err := time.LoadLocation(ReferenceLocale)
					if err != nil {
						validationError = err
						break
					}
					changeTime, err := time.Parse("2006-01-02 15:04", inputFields[key])
					if err != nil {
						validationError = err
						break
					}
					parsedLocaleTime := changeTime.In(referenceLocation).String()
					localeTimeTokens := strings.Split(parsedLocaleTime, " ")
					if len(localeTimeTokens) != 4 {
						validationError = errors.New(fmt.Sprintf("Can't parse locale time: %s", parsedLocaleTime))
						break
					}
					offset := localeTimeTokens[2]
					ticket.Fields.Unknowns[fieldID] = fmt.Sprintf("%sT%s:00.0%s", tokens[0], tokens[1], offset)
				}
			case "Change Type", "Change Risk":
				// customfield type = option
				ticket.Fields.Unknowns[fieldID] = jira.Option{Value: inputFields[key]}
			case "Reporter":
				// field type = User object
				user, _, err := j.jiraClient.User.Get(inputFields[key])
				if err != nil {
					log.Printf("Unable to set user to %s due to error: %s", inputFields[key], err.Error())
				} else {
					ticket.Fields.Reporter = user
				}
			case "Platform":
				// field type = label. This is custom handling for Core Video Player Team. Change Request tickets
				// don't actually have a "Platform" field as it doesn't apply to other teams. As discussed, add the
				// information under "Labels". See QE-1493
			case "Labels":
				// field type = string slice. Trim leading/trailing spaces, and avoid anything with spaces within
				// to prevent automation from proliferating labels. Log a message when this happens.
				for _, label := range strings.Split(inputFields[key], ",") {
					if strings.Contains(label, " ") {
						log.Printf("Skipping label because it contains space(s): %s", label)
					} else {
						ticket.Fields.Labels = append(ticket.Fields.Labels, strings.TrimSpace(label))
					}
				}
			case "Self Approve?":
				// field type = multiselect. Only set to Yes if input is Yes. Do not set default value. See QE-1498
				if inputFields[key] == "Yes" {
					// The underlying jira rest API requires a root-level array, eg [{'n1': 'v1}, {'n2': 'v2}]
					// See https://developer.atlassian.com/server/jira/platform/jira-rest-api-examples/#editing-an-issue-examples
					approveFlag := tcontainer.MarshalMap{
						"value": inputFields[key],
					}
					// However, go-jira client doesn't have a data structure that represents arrays at the root level,
					// so, create one manually based on reference doc: https://github.com/golang/go/wiki/InterfaceSlice
					var interfaceSlice = make([]interface{}, len(approveFlag))
					interfaceSlice[0] = approveFlag
					ticket.Fields.Unknowns[fieldID] = tcontainer.TryConvertToMarshalMap(interfaceSlice, nil)
					// Explicitly set approver = reporter (otherwise, jira sets value = user linked to oauth credential)
					user, _, err := j.jiraClient.User.Get(inputFields["Reporter"])
					if err != nil {
						log.Printf("Unable to set approver to %s due to error: %s", inputFields["Reporter"], err.Error())
					} else {
						ticket.Fields.Unknowns[JiraFields["Approver"]] = user
					}
				}
			case "Services Impacted":
				// field type = custom multiselect. Trim leading/trailing spaces
				refServices, err := NewServiceCatalogHelper().ListServices() // reference list from service catalog
				var validServices []string
				var invalidServices []string
				var inactiveServices []string

				if err != nil {
					break
				}

				for _, serviceName := range strings.Split(inputFields[key], ",") {
					sanitizedServiceName := strings.ToLower(strings.TrimSpace(serviceName))
					if sanitizedServiceName == "" {
						continue
					}
					foundMatchingRefServiceName := false
					for _, refServiceName := range refServices.Data.Services {
						if strings.EqualFold(refServiceName.Name, sanitizedServiceName) {
							if refServiceName.State == "Inactive" {
								inactiveServices = append(inactiveServices, refServiceName.Name) // matching reference name with inactive service
							} else {
								validServices = append(validServices, refServiceName.Name) // matching reference name with active service
							}
							foundMatchingRefServiceName = true
							break
						}
					}
					if !foundMatchingRefServiceName {
						invalidServices = append(invalidServices, serviceName) // raw user-input
					}
				}
				fieldID = ServicesImpacted[os.Getenv("ENVIRONMENT")]
				fieldValue := strings.Join(validServices, multiselectSeparator)
				ticket.Fields.Unknowns[fieldID] = fieldValue
				var errorMsg = ""
				if len(invalidServices) > 0 {
					errorMsg = "Invalid Services Impacted: " + strings.Join(invalidServices, ", ")
				}
				if len(inactiveServices) > 0 {
					errorMsg = errorMsg + "Inactive Services Impacted: " + strings.Join(inactiveServices, ", ")
				}
				if len(invalidServices) > 0 || len(inactiveServices) > 0 {
					validationError = errors.New(errorMsg)
				}
			case "Approver":
				if inputFields["Self Approve?"] != "Yes" { // if self-approve=Yes, explicitly set to reporter in "Self Approve" case above
					ticket.Fields.Unknowns[fieldID] = jira.User{Name: inputFields[key]}
					break
				}
			default:
				// customfield type = string
				ticket.Fields.Unknowns[fieldID] = inputFields[key]
			}
		}
	}
	return validationError
}

// GetIssue queries the specified issueId and returns Issue instance that is has a completely populated
// data-structure (eg Fields)
func (j *JiraHelper) GetIssue(issueID string) (*jira.Issue, error) {
	options := &jira.GetQueryOptions{
		// no custom options
	}
	issue, _, err := j.jiraClient.Issue.Get(issueID, options)

	return issue, err
}

// UpdateIssue updates the issue with specified inputFields
func (j *JiraHelper) UpdateIssue(issueID string, inputFields map[string]string) (*jira.Issue, error) {
	persistedTicket, err := j.GetIssue(issueID)
	if err != nil {
		return nil, LogAndReturnError(err, "Ticket to update doesn't exist: %s", issueID)
	}
	ticket := &jira.Issue{
		Key: issueID,
		Fields: &jira.IssueFields{
			// Twitch jira requires these 2 fields when updating any field
			Summary: persistedTicket.Fields.Summary,
			Type:    persistedTicket.Fields.Type,
		},
	}

	validationError := j.populateFieldsWithStructuredData(ticket, inputFields)
	// If there's local validation error, do not exit yet, but keep track of it.
	// Attempt to update the ticket with the validated inputFields.
	_, resp, err := j.jiraClient.Issue.Update(ticket)
	if err != nil {
		return nil, err
	}
	issue, getErr := j.GetIssue(issueID)
	if getErr != nil {
		// this shouldn't happen
		log.Printf("Unexpected scenario: unable to get issue due to: %s", getErr.Error())
		return nil, err
	}

	if validationError != nil {
		finalMessage := validationError.Error()
		if err != nil {
			// concat error messages from 2 sources (validation & update steps)
			message := j.GetResponseBodyString(resp)
			finalMessage = finalMessage + ". " + message
		}
		err = errors.New("Unable to update issue due to: " + finalMessage)
		return issue, err
	}

	if err != nil {
		message := j.GetResponseBodyString(resp)
		return issue, LogAndReturnError(err, "Unable to update issue due to: %s", message)
	}

	return issue, err
}

func (j *JiraHelper) LinkIssues(issueID string, linkedIssueIDs []string, linkType string) (*jira.Issue, error) {
	for _, linkedIssueID := range linkedIssueIDs {
		issueLink := &jira.IssueLink{
			Type: jira.IssueLinkType{
				Name: linkType,
			},
			InwardIssue: &jira.Issue{
				Key: issueID,
			},
			OutwardIssue: &jira.Issue{
				Key: linkedIssueID,
			},
		}
		resp, err := j.jiraClient.Issue.AddLink(issueLink)
		if err != nil {
			log.Printf("Received error while adding issue link: %s", err.Error())
			b, err := ioutil.ReadAll(resp.Body)
			if err != nil {
				log.Printf("Error while reading response body: %s", err.Error())
			} else {
				log.Printf("Response: %s", string(b))
			}
		}
	}
	return j.GetIssue(issueID)
}

type UserList []struct {
	Name        string `json:"name"`
	DisplayName string `json:"displayName"`
}

// FindUsers returns a list of users that match the search string.
// Reference implementation: https://github.com/andygrunwald/go-jira/blob/master/user.go
// Note: The REST API query parameter is named 'username' but it in fact matches the search string with ALL user properties
// JIRA API docs: https://developer.atlassian.com/cloud/jira/platform/rest/#api/2/
func (j *JiraHelper) FindUsers(searchString string) (*UserList, *jira.Response, error) {
	apiEndpoint := fmt.Sprintf("/rest/api/2/user/search?username=%s", searchString)

	req, err := j.jiraClient.NewRequest("GET", apiEndpoint, nil)
	if err != nil {
		return nil, nil, err
	}

	userList := new(UserList)
	resp, err := j.jiraClient.Do(req, userList)

	if err != nil {
		return nil, resp, err
	}
	return userList, resp, nil
}

// SanitizeErrorMessage improves readibility of raw messages from jira server. It replaces field IDs with field names
// and replaces commonly strings frequently reported as confusing by end-users.
func (j *JiraHelper) SanitizeErrorMessage(message string) string {
	for k, v := range JiraFields {
		message = strings.Replace(message, v, k, -1)
	}
	message = strings.Replace(message, "Option id 'null' is not valid", "Invalid value. See template for valid values.", -1)
	return message
}

// PostComment posts a comment on the specified issue
func (j *JiraHelper) PostComment(issueID string, text string) (*jira.Comment, error) {
	/* #nosec */
	_, err := j.GetIssue(issueID)
	if err != nil {
		return nil, err
	}

	c := &jira.Comment{
		Body: text,
	}
	comment, _, err := j.jiraClient.Issue.AddComment(issueID, c)
	return comment, err
}
