//nolint
package util

import (
	"github.com/nlopes/slack"
	"github.com/pkg/errors"
	"log"
	"os"
	"strings"
	"sync"
)

type SlackHelper struct {
	roleArn string
	client  *slack.Client
}

var (
	slackInstance *SlackHelper
	slackOnce     sync.Once
)

// GetSlackHelperInstance returns the singleton instance of SlackHelper.
// This uses reference implementation from http://marcio.io/2015/07/singleton-pattern-in-go/
func GetSlackHelperInstance() *SlackHelper {
	slackOnce.Do(func() {
		env := os.Getenv("ENVIRONMENT") // devtools' tooling on terraform + beanstalk provides this
		//Reference: roleArn should be the IAM role that has the sts:assumeRole policy
		//The role should be defined in terraform/twitch-cape-qe-aws/sandstorm_<env>.tf
		//The role should be allowed to access secret pattern: qa-eng/oml/<env>/*
		roleArn := "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/qa-eng-" + env + "-automation-webhooks"

		slackInstance = &SlackHelper{
			roleArn: roleArn,
		}

		accessToken := os.Getenv("SLACK_API_TOKEN")
		if len(accessToken) == 0 {
			accessTokenSS, err := GetSandstormHelperInstance().GetSecrets("qa-eng/automation-webhooks/" + env + "/slack-token")
			if err != nil {
				log.Fatalf("Hard failure: Unable to get slack API token from environment or sandstorm. Can't instantiate helper.")
			} else {
				accessToken = accessTokenSS
			}
		}
		slackInstance.client = slack.New(accessToken)
	})
	return slackInstance
}

// FindMatchingUser finds a user that matches the userId string in email address or user name fields
// userId is required. fullName is optional.
func (s *SlackHelper) FindMatchingUser(userId string, fullName string) (*slack.User, error) {
	// attempt to retrieve user info by inconsistent email address usage
	user, err := s.client.GetUserByEmail(userId + "@justin.tv")
	if err == nil {
		return user, nil
	}
	user, err = s.client.GetUserByEmail(userId + "@twitch.tv")
	if err == nil {
		return user, nil
	}
	// worst case -- fallback is to search through user list -- GetUserByEmail() can fail due to various errors
	users, err := s.client.GetUsers()
	if err != nil {
		return nil, err
	}
	for _, user := range users {
		if user.Name == userId || strings.HasPrefix(user.Profile.Email, userId+"@") {
			return &user, nil
		}
		if len(fullName) > 0 && strings.ToLower(user.RealName) == strings.ToLower(fullName) {
			return &user, nil
		}
	}
	return nil, errors.New("Could not find user by id or name")
}

// DirectMessage sends a direct message to target user
func (s *SlackHelper) DirectMessage(user *slack.User, message string) error {
	_, _, channelID, err := s.client.OpenIMChannel(user.ID)
	if err != nil {
		log.Printf("Could not create IM channel for targetUser %s due to error: %s", user.Name, err.Error())
		return err
	}

	postParams := slack.PostMessageParameters{
		AsUser: true,
	}
	_, _, err = s.client.PostMessage(channelID, message, postParams)

	if err != nil {
		log.Printf("Could not post direct message to targetUser %s due to error: %s", user.Name, err.Error())
	}
	return err
}
