//nolint
package webhook

/* #nosec */
import (
	"code.justin.tv/qe/automation-webhooks/object"
	t "code.justin.tv/qe/automation-webhooks/task"
	"code.justin.tv/qe/automation-webhooks/util"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"regexp"
	"strings"
	"time"
)

const (
	EVENTHEADER         = "X-Github-Event"
	PULLREQUEST         = "pull_request"
	COMMITCOMMENT       = "commit_comment"
	ISSUECOMMENT        = "issue_comment"
	PRCOMMENT           = "pull_request_review_comment"
	BRANCHCREATE        = "create"
	RELEASE             = "release"
	PING                = "ping"
	BOTNAME             = "changerequest"
	AUTORESPONSE_PREFIX = "Automated Response: "
	MSG_FAIL            = AUTORESPONSE_PREFIX + `Your PR comment failed to create a Change Request ticket.
Please check that your PR contains the mandatory fields for a Change Request ticket and try again.
FYI, the error was: %s`
	MSG_SUCCESS = AUTORESPONSE_PREFIX + `Your PR comment triggered a Change Request.
Please refer to %s/browse/%s for more information`
	MSG_SUCCESS_JIRA_KEY_REGEX = AUTORESPONSE_PREFIX + `Your PR comment triggered a Change Request.\nPlease refer to (.*)\/browse\/([a-zA-Z]{1,}-[0-9]{1,}) for more information`
	MSG_INCOMPLETE_GITHUB      = AUTORESPONSE_PREFIX + `Your PR comment triggered an incomplete Change Request.
FYI, the error was: %s.

:warning: Please review and complete your CR [%s](%s/browse/%s)`
	MSG_INCOMPLETE_SLACK = AUTORESPONSE_PREFIX + `Your PR comment triggered an incomplete Change Request.
FYI, the error was: %s. 

:warning: Please review and complete your CR <%s/browse/%s|%s>`
	JIRA_COMMENT_PREFIX    = "Automated Message: "
	JIRA_COMMENT_PR_CLOSED = JIRA_COMMENT_PREFIX + `The associated PR at https://git.xarth.tv/%s/pull/%d has been closed`
	JIRA_COMMENT_PR_LINKED = JIRA_COMMENT_PREFIX + `The PR at https://git.xarth.tv/%s/pull/%d is now part of this CR. 

%s`
	MSG_FAIL_LINK_PR    = AUTORESPONSE_PREFIX + `Failed to link this PR to existing CR %s due to: %s`
	MSG_MISSING_CR      = AUTORESPONSE_PREFIX + `Failed to link this PR to existing CR because the CR ticket is not specified`
	MSG_NONEXIST_CR     = AUTORESPONSE_PREFIX + `Failed to link this PR to the specified CR because the CR ticket does not exist`
	MSG_NOT_CR          = AUTORESPONSE_PREFIX + `Failed to link this PR to existing CR https://jira.twitch.com/browse/%s because the specified ticket is not a Change Request ticket`
	MSG_CLOSED_CR       = AUTORESPONSE_PREFIX + `Failed to link this PR to the CR https://jira.twitch.com/browse/%s because it is not in the Open state`
	MSG_EXPIRED_CR      = AUTORESPONSE_PREFIX + `Failed to link this PR to the CR https://jira.twitch.com/browse/%s because its Change End Date/Time is in the past`
	MSG_SUCCESS_LINK_PR = AUTORESPONSE_PREFIX + `Successfully linked this PR to existing CR https://jira.twitch.com/browse/%s`
)

// ChangeRequestHandler will contain the handler that is the entrypoint to the webhook and codifies the workflow automation
// at a high level
type ChangeRequestHandler struct{}

func (c *ChangeRequestHandler) getHttpRequestBodyString(r *http.Request) (string, error) {
	bodyBytes, err := ioutil.ReadAll(r.Body)
	var bodyString string
	if err == nil {
		bodyString = string(bodyBytes[:])
	}
	return bodyString, err
}

func (c *ChangeRequestHandler) handleFailure(w http.ResponseWriter, err error, message string, statusCode int) {
	log.Println(err, message)
	http.Error(w, message, statusCode)
}

func (c *ChangeRequestHandler) Handler(w http.ResponseWriter, r *http.Request) {
	eventType := r.Header.Get(EVENTHEADER)
	log.Printf("Incoming event type: %s", eventType)
	bodyBytes, err := ioutil.ReadAll(r.Body)
	if err != nil {
		c.handleFailure(w, err, "Unable to read incoming http request body", http.StatusInternalServerError)
		return
	}

	var task t.ChangeRequestInterface
	switch eventType {
	case ISSUECOMMENT:
		// test event: d3091190-11e5-11e8-87eb-dcf75d3a5e78
		var comment object.Comment
		err = json.Unmarshal(bodyBytes, &comment)
		if err != nil {
			c.handleFailure(w, err, "Unable to unmarshal comment", http.StatusInternalServerError)
			return
		}
		// NoOp operations
		if comment.Action == "deleted" {
			log.Printf("User deleted PR review comment %d in repo %s", comment.Comment.ID, comment.Repository.FullName)
			return // not a problem. just don't process it further
		}
		if comment.Comment.Body == t.PrTemplate {
			// plain vanilla comment just updated by the webhook itself
			log.Printf("Skip further processing of comment just updated with PR template")
			return // not a problem. just don't process it further
		}
		task, err = t.NewPRTriggeredChangeRequestTask(&comment)
		if err != nil {
			// this really shouldn't happen
			log.Printf("Unable to initialize task due to %s", err.Error())
			c.handleFailure(w, err, "Unable to unmarshal comment", http.StatusInternalServerError)
			return
		}
		//if trigger word, could create cr if fields are filled in or comment is updated, if no cr thing exists, add template
		sanitizedCommentBody := strings.TrimSpace(strings.ToLower(comment.Comment.Body))
		triggerNewCR := strings.HasPrefix(sanitizedCommentBody, "create cr") ||
			strings.Contains(comment.Comment.Body, t.ChangeRequestInfoTitle)
		if triggerNewCR {
			issue, err := task.ActOnTrigger()
			issueCreated := issue != nil
			errorOccured := err != nil
			issueNotCreatedTemplateAddedToCR := !issueCreated && !errorOccured
			partiallyCreatedIssue := errorOccured && issueCreated
			issueNotCreateDueToError := errorOccured && !issueCreated
			issueCreatedCompletely := !errorOccured && issueCreated
			jira := util.GetJiraHelperInstance()
			switch {
			case issueCreatedCompletely:
				// C. happy path - issue created completely
				message := jira.SanitizeErrorMessage(fmt.Sprintf(MSG_SUCCESS, jira.GetJiraURL(), issue.Key))
				w.WriteHeader(http.StatusCreated)
				c.postSlackMessage(comment.Sender.Login, message)
				c.postPRReviewComment(BOTNAME, comment.Repository.FullName, comment.PullRequest.Number, message)
				return
			case issueNotCreatedTemplateAddedToCR:
				// A. issue not created. template was added to CR. not an error
				return
			case partiallyCreatedIssue:
				// D. partially created issue
				slackMsg := jira.SanitizeErrorMessage(fmt.Sprintf(MSG_INCOMPLETE_SLACK, err.Error(), jira.GetJiraURL(),
					issue.Key, issue.Key))
				prMsg := jira.SanitizeErrorMessage(fmt.Sprintf(MSG_INCOMPLETE_GITHUB, err.Error(), issue.Key,
					jira.GetJiraURL(), issue.Key))
				c.postSlackMessage(comment.Sender.Login, slackMsg)
				c.postPRReviewComment(BOTNAME, comment.Repository.FullName, comment.PullRequest.Number, prMsg)
				c.handleFailure(w, err, "Payload accepted, but incomplete CR created", http.StatusAccepted)
				return
			case issueNotCreateDueToError:
				// B. issue not created due to some error
				message := util.GetJiraHelperInstance().SanitizeErrorMessage(fmt.Sprintf(MSG_FAIL, err.Error()))
				c.postSlackMessage(comment.Sender.Login, message)
				c.postPRReviewComment(BOTNAME, comment.Repository.FullName, comment.PullRequest.Number, message)
				c.handleFailure(w, err, "Payload accepted, but unable to to create CR", http.StatusAccepted)
				return
			}
		}

		triggerLinkPRToExistingCR := strings.HasPrefix(sanitizedCommentBody, "link cr")
		if triggerLinkPRToExistingCR {
			c.handleLinkCRComment(&comment, w)
		}
	case BRANCHCREATE:
		log.Println("TODO: automation triggered by branch creation")
		http.Error(w, "This event type is not handled at the moment", http.StatusNotImplemented)
	case RELEASE:
		var ghRelease object.GithubRelease
		err = json.Unmarshal(bodyBytes, &ghRelease)
		if err != nil {
			c.handleFailure(w, err, "Unable to unmarshal github release", http.StatusInternalServerError)
			return
		}
		task, err = t.NewReleaseTriggeredChangeRequestTask(&ghRelease)
		if err != nil {
			// this really shouldn't happen
			log.Printf("Unable to initialize task due to %s", err.Error())
			c.handleFailure(w, err, "Unable to initialize task", http.StatusInternalServerError)
			return
		}
		triggerCRAutomation := strings.Contains(ghRelease.Release.Body, t.ChangeRequestInfoTitle)
		if triggerCRAutomation {
			issue, err := task.ActOnTrigger()
			issueCreated := issue != nil
			errorOccured := err != nil
			invalidCase := !issueCreated && !errorOccured
			partiallyCreatedIssue := errorOccured && issueCreated
			issueNotCreateDueToError := errorOccured && !issueCreated
			issueCreatedCompletely := !errorOccured && issueCreated
			switch {
			case issueCreatedCompletely:
				// C. happy path - issue created completely. (scenario A doesn't happen for Release-triggerred automation)
				message := fmt.Sprintf(MSG_SUCCESS, util.GetJiraHelperInstance().GetJiraURL(), issue.Key)
				w.WriteHeader(http.StatusCreated)
				c.postSlackMessage(ghRelease.Sender.Login, message)
				return
			case partiallyCreatedIssue:
				// D. partially created issue
				message := fmt.Sprintf(MSG_INCOMPLETE_SLACK, err.Error(), util.GetJiraHelperInstance().GetJiraURL(), issue.ID)
				c.postSlackMessage(ghRelease.Sender.Login, message)
				c.handleFailure(w, err, "Payload accepted, but incomplete CR created", http.StatusAccepted)
				return
			case issueNotCreateDueToError:
				// B. issue not created due to some error
				message := fmt.Sprintf(MSG_FAIL, err.Error())
				c.postSlackMessage(ghRelease.Sender.Login, message)
				c.handleFailure(w, err, "Payload accepted, but unable to to create CR", http.StatusAccepted)
				return
			case invalidCase:
				// A. should be impossible to get here
				log.Println("Encountered unexpected scenario: issue not created, BUT error is nil")
			}
		}
	case PING:
		var ping object.Ping
		err = json.Unmarshal(bodyBytes, &ping)
		log.Printf("CR automation webhook (%s) enabled by %s", ping.Hook.Type, ping.Sender.Login)
	case PULLREQUEST:
		var pullRequest object.PullRequest
		err = json.Unmarshal(bodyBytes, &pullRequest)
		if err != nil {
			c.handleFailure(w, err, "Unable to unmarshal pull request", http.StatusInternalServerError)
			return
		}
		err = c.handlePullRequestEvent(&pullRequest)
		if err != nil {
			c.handleFailure(w, err, "Error processing pull request closure", http.StatusInternalServerError)
			return
		}
	default:
		log.Println("This event type is not handled at the moment")
		http.Error(w, "This event type is not handled at the moment", http.StatusNotImplemented)
	}
}

func (c *ChangeRequestHandler) handleLinkCRComment(comment *object.Comment, w http.ResponseWriter) {
	sanitizedCommentBody := strings.TrimSpace(strings.ToLower(comment.Comment.Body))
	// Link additional PRs to existing CR ticket by posting a new comment on the ticket
	tokens := strings.Split(sanitizedCommentBody, " ")
	if len(tokens) < 3 {
		// complain that CR ticket was not provided
		c.postSlackMessage(comment.Sender.Login, MSG_MISSING_CR)
		c.postPRReviewComment(BOTNAME, comment.Repository.FullName, comment.PullRequest.Number, MSG_MISSING_CR)
		return
	}
	crTicketId := strings.ToUpper(tokens[2])
	// perform validation
	jira := util.GetJiraHelperInstance()
	crTicket, err := jira.GetIssue(crTicketId)
	if err != nil {
		// complain that CR ticket does not exist
		c.postSlackMessage(comment.Sender.Login, MSG_NONEXIST_CR)
		c.postPRReviewComment(BOTNAME, comment.Repository.FullName, comment.PullRequest.Number, MSG_NONEXIST_CR)
		return
	}
	// check type = change request
	if crTicket.Fields.Type.Name != "Change Request" {
		// complain that specified ticket is not a CR
		message := fmt.Sprintf(MSG_NOT_CR, crTicketId)
		c.postSlackMessage(comment.Sender.Login, message)
		c.postPRReviewComment(BOTNAME, comment.Repository.FullName, comment.PullRequest.Number, message)
		return
	}
	// check cr is open
	if crTicket.Fields.Status.Name != "Open" {
		// complain that CR ticket does not exist
		message := fmt.Sprintf(MSG_CLOSED_CR, crTicketId)
		c.postSlackMessage(comment.Sender.Login, message)
		c.postPRReviewComment(BOTNAME, comment.Repository.FullName, comment.PullRequest.Number, message)
		return
	}
	// check end date is active (current or future)
	changeStartDateStr := crTicket.Fields.Unknowns[util.JiraFields["Change End Date/Time"]].(string)
	dateTokens := strings.Split(changeStartDateStr, ".")
	shortStartDateStr := dateTokens[0]
	referenceLocation, err := time.LoadLocation(util.ReferenceLocale)
	if err != nil {
		c.postSlackMessage(comment.Sender.Login, err.Error())
		c.postPRReviewComment(BOTNAME, comment.Repository.FullName, comment.PullRequest.Number, err.Error())
		c.handleFailure(w, err, "Payload accepted, but unable to to create CR due to server error", http.StatusAccepted)
		return
	}
	changeTime, err := time.Parse("2006-01-02T15:04:05", shortStartDateStr)
	parsedLocaleTime := changeTime.In(referenceLocation)
	if parsedLocaleTime.Before(time.Now()) {
		// complain that CR ticket is not in the future
		message := fmt.Sprintf(MSG_EXPIRED_CR, crTicketId)
		c.postSlackMessage(comment.Sender.Login, message)
		c.postPRReviewComment(BOTNAME, comment.Repository.FullName, comment.PullRequest.Number, message)
		return
	}

	commentText := fmt.Sprintf(JIRA_COMMENT_PR_LINKED, comment.Repository.FullName, comment.PullRequest.Number, comment.PullRequest.Body)
	_, err = jira.PostComment(crTicketId, commentText)
	if err != nil {
		message := fmt.Sprintf(MSG_FAIL_LINK_PR, crTicketId, err.Error())
		c.postSlackMessage(comment.Sender.Login, message)
		c.postPRReviewComment(BOTNAME, comment.Repository.FullName, comment.PullRequest.Number, message)
		c.handleFailure(w, err, "Payload accepted, but unable to to create CR due to Jira api error", http.StatusAccepted)
	} else {
		message := fmt.Sprintf(MSG_SUCCESS_LINK_PR, crTicketId)
		c.postSlackMessage(comment.Sender.Login, message)
		c.postPRReviewComment(BOTNAME, comment.Repository.FullName, comment.PullRequest.Number, message)
	}
}

func (c *ChangeRequestHandler) handlePullRequestEvent(pullRequest *object.PullRequest) error {
	if pullRequest.Action == "closed" { // when PR is closed
		// update jira ticket
		githubHelper := &util.GithubHelper{}
		githubHelper.Init()
		comments, _, err := githubHelper.ListPRComments(pullRequest.Repository.FullName, pullRequest.Number)
		if err != nil {
			return err
		}
		// find the JIRA ticket generated by the webhook from the PR comments using a strict regex
		r := regexp.MustCompile(MSG_SUCCESS_JIRA_KEY_REGEX)
		matchingCommentProcessed := false
		for _, comment := range comments {
			matches := r.FindAllStringSubmatch(comment.GetBody(), -1) // there could be more than 1 ticket. update them all
			for _, group := range matches {
				if len(group) == 3 { // 2 groups in the regex + 1 built-in group for entire string match
					jiraTicketKey := group[2]
					jira := util.GetJiraHelperInstance()
					// setup comment text
					comment := fmt.Sprintf(JIRA_COMMENT_PR_CLOSED, pullRequest.Repository.FullName, pullRequest.Number)
					_, err := jira.PostComment(jiraTicketKey, comment)
					if err != nil {
						return err
					}
					matchingCommentProcessed = true
				}
			}
		}
		if matchingCommentProcessed {
			log.Printf("found associated change request ticket(s) and posted comment")
		}
	}
	return nil
}

// post PR review comment (updates existing comment if found)
func (c *ChangeRequestHandler) postPRReviewComment(userName string, fullRepoName string, prNumber int, text string) {
	githubHelper := &util.GithubHelper{}
	githubHelper.Init()
	comments, _, err := githubHelper.ListPRComments(fullRepoName, prNumber)
	if err != nil {
		log.Printf("Error listing PR review comments on %s PR #%d: %s", fullRepoName, prNumber, err.Error())
		return
	}

	lastComment := comments[len(comments)-1]
	isBotComment := *lastComment.User.Login == BOTNAME && strings.HasPrefix(*lastComment.Body, AUTORESPONSE_PREFIX)
	if isBotComment {
		// update the existing bot comment
		_, _, err = githubHelper.UpdatePRReviewComment(fullRepoName, int(*lastComment.ID), text)
	} else {
		// create new bot comment
		_, _, err = githubHelper.CreatePRReviewComment(fullRepoName, prNumber, text)
	}
	if err != nil {
		log.Printf("Error posting PR review comment on %s PR #%d: %s", fullRepoName, prNumber, err.Error())
	}
}

// post slack message to a user that is mapped to the github user
func (c *ChangeRequestHandler) postSlackMessage(userName string, text string) {
	githubHelper := &util.GithubHelper{}
	githubHelper.Init()
	githubUser, _, err := githubHelper.GetUser(userName)
	if err != nil {
		log.Printf("Error finding github user %s", userName)
	}
	slack := util.GetSlackHelperInstance()
	slackUser, err := slack.FindMatchingUser(*githubUser.Login, *githubUser.Name)
	if err != nil {
		log.Printf("Error finding slack user %s", userName)
	}
	slack.DirectMessage(slackUser, text)
}
