""" Aws device Farm client wrapper for execution """
import logging
import os
import time

import boto3

from devicefarmutils.aws_device_farm_exception import AwsDeviceFarmException
from devicefarmutils.aws_test_results import TestResults


class AWSDeviceFarm(object):
    """
    Uploads and Executes on device farm using boto3 api
    """

    # Project constants
    PROJECT_TYPE = 'APPIUM_WEB_PYTHON'
    PROJECT_TEST_PACKAGE = 'APPIUM_WEB_PYTHON_TEST_PACKAGE'
    PACKAGE_NAME = 'test_bundle.zip'
    DEFAULT_WAIT_TIME = 30
    REGION = 'us-west-2'
    SERVICE_NAME = 'devicefarm'
    TEST_RUN_NAME = 'mobile-web-device-tests'

    # Creates an instance of device farm
    def __init__(self, aws_key_id, aws_secret_key, device_farm_arn):
        self.client = boto3.client(self.SERVICE_NAME, region_name=self.REGION,
                                   aws_access_key_id=aws_key_id,
                                   aws_secret_access_key=aws_secret_key)
        self.device_farm_arn = device_farm_arn
        self.test_results = TestResults(self.client)
        logging.basicConfig(format='%(asctime)s %(message)s')
        self.logger = logging.getLogger(__name__)
        self.logger.setLevel(logging.INFO)

    def package_and_execute(self, project_name, device_pool_name):
        """
        Executes the tests against a device pool on aws device farm
        :param project_name: name of the project
        :param device_pool_name:  device pool name
        :return:
        """
        self.package_tests()
        project_arn = self.select_project(project_name)
        upload_arn = self.upload_package(project_arn)
        self.check_upload_status(upload_arn)
        device_arn = self.select_device_pool(project_arn, device_pool_name)
        run_arn = self.schedule_run(project_arn, device_arn, upload_arn)
        self.wait_for_run_completion(run_arn)

    def select_project(self, project_name):
        """
        get project arn based on project name
        :param project_name:
        :return:
        """
        projects_data = self.client.list_projects(arn=self.device_farm_arn)
        all_projects = projects_data['projects']
        return self.get_arn_by_name(all_projects, project_name)

    #
    def select_device_pool(self, project_arn, device_pool_name):
        """
        Gets device_pool_arn on which tests will be executed
        :param project_arn:
        :param device_pool_name:
        :return:
        """
        device_pool_data = self.client.list_device_pools(arn=project_arn)
        all_device_pools = device_pool_data['devicePools']
        return self.get_arn_by_name(all_device_pools, device_pool_name)

    def upload_package(self, project_arn):
        """
        Uploads test_bundle.zip for the selected project_arn
        :param project_arn:
        :return:
        """
        upload_package = self.client.create_upload(projectArn=project_arn,
                                                   name=self.PACKAGE_NAME,
                                                   type=self.PROJECT_TEST_PACKAGE)
        upload_arn = upload_package['upload']['arn']
        upload_url = upload_package['upload']['url']
        # Create curl upload command
        self.logger.info('Uploading the tests package using curl')
        upload_cmd = 'curl -T test_bundle.zip "' + upload_url + '" -v'
        os.system(upload_cmd)
        return upload_arn

    def check_upload_status(self, upload_arn):
        """
        Checks the upload status
        :param upload_arn:
        :return:
        """
        timeout = time.time() + self.DEFAULT_WAIT_TIME
        upload_status = False
        while time.time() < timeout:
            upload_result = self.client.get_upload(arn=upload_arn)['upload']
            if upload_result['status'] in ['SUCCEEDED', 'FAILED']:
                upload_status = True
                break
            time.sleep(1)
        if not upload_status:
            raise AwsDeviceFarmException("upload action timed out after " +
                                         str(self.DEFAULT_WAIT_TIME))
        if upload_result['status'] == "Failed":
            raise AwsDeviceFarmException("upload action was unsuccessful")
        self.logger.info("Test package uploaded successfully")

    def schedule_run(self, project_arn, device_pool_arn, upload_arn):
        """
        Schedules run on selected project and device pool
        :param project_arn:
        :param device_pool_arn:
        :param upload_arn:
        :return:
        """
        # Schedule AWS APP Project
        self.logger.info('Schedule the test run on AWS device farm')
        scheduler = self.client.schedule_run(projectArn=project_arn, devicePoolArn=device_pool_arn,
                                             name=self.TEST_RUN_NAME, test={'type': self.PROJECT_TYPE,
                                                                            'testPackageArn': upload_arn})
        return scheduler['run']['arn']

    def wait_for_run_completion(self, run_arn):
        """
        Waits for the test run to complete and fetch results
        :param run_arn:
        :return:
        """
        self.logger.info('Please wait while the job is executing')
        while not self.client.get_run(arn=run_arn)['run']['status'] == 'COMPLETED':
            time.sleep(self.DEFAULT_WAIT_TIME)
        # Print the summary of results
        self.logger.info("Test execution completed")
        self.test_results.get_results_summary(run_arn)

    def get_arn_by_name(self, data_list, name):
        self.logger.info('Get the ARN for ' + name)
        index = -1
        for i in range(0, len(data_list)):
            if data_list[i]['name'] == name:
                index = i
        if index == -1:
            raise AwsDeviceFarmException("ERROR : " + name + " Not found")
        return data_list[index]['arn']

    def package_tests(self):
        """
         Package the tests package into a zip folder.
        :return:
        """
        self.logger.info("packaging test package into a zip")
        os.system('rm -rf wheelhouse')
        os.system('rm -rf test_bundle.zip')
        os.system('py.test --collect-only tests/')

        os.system('find . -name \'__pycache__\' -type d -exec rm -r {} +')
        os.system('find . -name \'*.pyc\' -exec rm -f {} +')
        os.system('find . -name \'*.pyo\' -exec rm -f {} +')
        os.system('find . -name \'*~\' -exec rm -f {} +')

        os.system('pip wheel --wheel-dir wheelhouse -r requirements.txt')

        # Zip tests/, wheelhouse/, and requirements.txt into test_bundle.zip
        os.system('zip -r test_bundle.zip tests/ wheelhouse/ requirements.txt')
