""" Sanitize the test results from aws device farm exection"""
import os
from _elementtree import parse

import requests

from devicefarmutils.aws_device_farm_exception import AwsDeviceFarmException
from xunitmerge.xmerge import merge_xunit


class TestResults(object):
    """Format the test results from aws device farm exection"""

    SKIPPED_RESULT = ['ERRORED', 'SKIPPED', 'PENDING']

    def __init__(self, client):
        self.client = client
        self.total_count = 0
        self.pass_count = 0
        self.fail_count = 0
        self.skip_count = 0
        self.error_count = 0
        self.skip_job = 0
        self.results_output_folder = os.environ['test_results_folder']


    def get_results_summary(self, run_arn):
        """
        print test execution summary
        :param run_arn:
        :return:
        """
        run_result = self.client.get_run(arn=run_arn)['run']
        print '\n###### Test Execution for ' + run_result['name'] + ':' + run_result['result'] + \
              ' #######'
        self.get_jobs_results(run_arn)
        self.merge_xml(self.results_output_folder)
        print '\n************ Test Results Summary ************'
        print '\nTotal: ' + str(self.total_count) + '  Failed: ' + str(self.fail_count) + \
              '  Passed: ' + str(self.total_count - self.fail_count)

        print '\nJobs Skipped: ' + str(self.skip_job)
        print '\n**********************************************'

        if self.skip_job > 0 or self.fail_count > 0:
            raise AwsDeviceFarmException("***** Test Failures Found *********")

    def get_jobs_results(self, run_arn):
        """
        Get the list of job results executed
        :param run_arn:
        :return:
        """
        jobs_list = self.client.list_jobs(arn=run_arn)['jobs']
        for job_result in jobs_list:
            jobs_result_status = job_result['result']
            job_name = job_result['name']
            print '************************************************************************'
            print 'Device Name: ' + job_name + ': ' + jobs_result_status
            if jobs_result_status in TestResults.SKIPPED_RESULT:
                print ' -> Error Message:' + 'Execution was skipped due to device not available'
                self.skip_job = self.skip_job + 1
            suites_list = self.get_suites(job_result['arn'])
            for suite in suites_list:
                self.get_tests_results(suite, job_result['device']['name'])

    def get_tests_results(self, suite, device_name):
        """
        Get results for per suite per device
        :param suite:
        :param device_name:
        :return:
        """
        test_list = self.client.list_tests(arn=suite['arn'])['tests']
        for test in test_list:
            print test['name'] + ': ' + test['result']
            results_folder = self.results_output_folder + '/' + device_name + '/' + test['name']
            if test['result'] == 'FAILED':
                self.fail_count = self.fail_count + 1
                results_folder = self.results_output_folder + '/failed/' + device_name + '/' + test['name']
                print ' -> Error Message: ' + test['message']
            if not test['name'] in ['Setup Test', 'Teardown Test'] and not test['result'] in TestResults.SKIPPED_RESULT:
                self.total_count = self.total_count + 1
                artifacts_list = self.get_artifacts(test['arn'], 'FILE')
                screenshots_list = self.get_artifacts(test['arn'], 'SCREENSHOT')
                self.download_artifacts(test['name'], device_name, artifacts_list, results_folder)
                self.download_artifacts(test['name'], device_name, screenshots_list, results_folder)

    def get_suites(self, job_arn):
        """
         Get list of suites executed per job/device
        :param job_arn:
        :return:
        """
        suites_list = self.client.list_suites(arn=job_arn)['suites']
        return suites_list

    def get_unique_problems(self, run_arn):
        """
        Get unique problems/failures per run
        :param run_arn:
        :return:
        """
        failed_list = self.client.list_unique_problems(arn=run_arn)['uniqueProblems']['FAILED']
        if failed_list:
            print '******************** Unique Problems ********************'
            for errors in failed_list:
                errors = errors['problems']
                for error in errors:
                    print error['device']['name'] + ': ' + error['test']['name']
                    print ' -> Error Message: ' + error['message']

    def get_artifacts(self, test_arn, artifact_type):
        """
        Get list of artifacts based on type for a test
        :param test_arn:
        :param artifact_type:
        :return:
        """
        return self.client.list_artifacts(arn=test_arn, type=artifact_type)['artifacts']

    @staticmethod
    def download_artifacts(test_name, device_name, artifact_list, download_folder):
        """
        download the artifacts
        :param test_name:
        :param device_name:
        :param artifact_list:
        :return:
        """
        test_name = test_name.replace(' ', '_')

        if not os.path.exists(download_folder):
            os.makedirs(download_folder)
        for artifact in artifact_list:
            response = requests.get(artifact['url'], allow_redirects=True)
            if artifact['name'] == 'Appium Python XML Output':
                filepath = download_folder + '/' + test_name + '.xml';
                open(filepath, 'wb').write(response.content)
                TestResults.append_device_name_to_classname(filepath, device_name)
            elif artifact['name'] == 'Video':
                open(download_folder + '/' + test_name + '.mp4', 'wb').write(response.content)
            elif artifact['type'] == 'SCREENSHOT':
                screenshot_folder = download_folder + '/' + 'screenshots'
                if not os.path.exists(screenshot_folder):
                    os.makedirs(screenshot_folder)
                open(screenshot_folder + '/' + artifact['name'].replace(' ', '_')
                     + '.png', 'wb').write(response.content)

    @staticmethod
    def append_device_name_to_classname(filename, device_name):
        """
        Append device name to classname for better debugging in test results
        :param filename:
        :param device_name:
        :return:
        """
        device_name = device_name.replace(' ', '_')
        doc = parse(filename)
        root = doc.getroot()
        element = root.find('testcase')
        name = element.get('classname')
        element.set('classname', name + '_' + device_name)
        doc.write(filename, xml_declaration=True)

    @staticmethod
    def merge_xml(output_folder):
        """
        Merge xml's into a single xml
        :return:
        """
        files_list = []
        for root, dirs, files in os.walk(output_folder):
            for filename in files:
                if filename.endswith('.xml'):
                    files_list.append(os.path.join(root, filename))
        merged_xml = output_folder + '/merged.xml'
        merge_xunit(files=files_list, output=merged_xml)
