package config

import (
	"fmt"
	"github.com/sirupsen/logrus"
	"github.com/sirupsen/logrus/hooks/test"
	"log"
	"os"
	"strings"
)

const (
	logLevelEnvName = "LOG_LEVEL"
	targetInstanceTypeEnvName = "GRID_TARGET_INSTANCE_TYPE"
)

// App is a global configuration object for the application
type App struct {
	Logger             *logrus.Logger
	TargetInstanceType string
}

// Creates a New Application Config
func NewAppConfig() (*App, error) {
	targetInstanceType, err := getTargetInstanceType()
	if err != nil {
		return nil, err
	}

	appConfig := &App{
		Logger: newLogger(),
		TargetInstanceType: targetInstanceType,
	}
	return appConfig, nil
}

// Gets the user specified Target Instance Type, which the user sets via env var: GRID_TARGET_INSTANCE_TYPE
// Returns an error if the environment variable was not provided
func getTargetInstanceType() (string, error) {
	targetInstanceType := os.Getenv(targetInstanceTypeEnvName)
	if targetInstanceType == "" {
		return "", fmt.Errorf("must provide ENV Variable: %s", targetInstanceTypeEnvName)
	}
	return targetInstanceType, nil
}

// Returns a log level to use based on the environment variable logLevelEnvName
// No input will default to Info
// Unrecognized input will return Info
func getLogLevel() logrus.Level {
	var logLevel logrus.Level

	// toLower - case insensitive input
	switch requestedLevel := strings.ToLower(os.Getenv(logLevelEnvName)); requestedLevel {
	case "debug":
		logLevel = logrus.DebugLevel
	case "info":
		logLevel = logrus.InfoLevel
	case "warn", "warning":
		logLevel = logrus.WarnLevel
	case "error":
		logLevel = logrus.ErrorLevel
	case "fatal":
		logLevel = logrus.FatalLevel
	case "":
		logLevel = logrus.InfoLevel
	default:
		log.Printf("Did not recognize log level: %s. Using INFO\n", requestedLevel)
		logLevel = logrus.InfoLevel
	}
	return logLevel
}

// Returns a new logger object with color
func newLogger() *logrus.Logger {
	logger := logrus.New()
	logger.SetLevel(getLogLevel())
	logger.SetFormatter(&logrus.TextFormatter{
		ForceColors:   true,
		DisableColors: false,
		FullTimestamp: false, // jenkins job will output timestamps...
	})
	return logger
}

// Returns a mock version of AppConfig - used only for testing!
func NewAppConfigMock() *App {
	logger, _ := test.NewNullLogger()
	return &App{
		Logger:             logger,
		TargetInstanceType: "c4.xlarge",
	}
}
