package config

import (
	"fmt"
	"github.com/sirupsen/logrus"
	"github.com/stretchr/testify/assert"
	"os"
	"testing"
)

func TestNewAppConfig(t *testing.T) {
	t.Run("when no instance type returns an error", func (t *testing.T) {
		config, err := NewAppConfig()
		assert.EqualError(t, err, fmt.Sprintf("must provide ENV Variable: %s", targetInstanceTypeEnvName))
		assert.Nil(t, config)
	})

	t.Run("when providing an instance type, returns a config", func (t *testing.T) {
		err := os.Setenv(targetInstanceTypeEnvName, "test")
		defer os.Unsetenv(targetInstanceTypeEnvName)
		if err != nil {
			t.Fatalf("encountered err: %v", err)
		}

		config, err := NewAppConfig()
		assert.NoError(t, err)
		assert.NotNil(t, config)
	})

	t.Run("sets log level properly", func (t *testing.T) {
		var logLevelTests = []struct {
			in string
			out logrus.Level
		}{
			{"", logrus.InfoLevel },
			{"DEBUG", logrus.DebugLevel},
			{"INFO", logrus.InfoLevel},
			{"WARN", logrus.WarnLevel},
			{"ERROR", logrus.ErrorLevel},
			{"FATAL", logrus.FatalLevel},
		}

		err := os.Setenv(targetInstanceTypeEnvName, "test")
		defer os.Unsetenv(targetInstanceTypeEnvName)
		if err != nil {
			t.Fatalf("encountered err: %v", err)
		}

		for _, tt := range logLevelTests {
			t.Run(tt.in, func (t *testing.T) {
				err := os.Setenv(logLevelEnvName, tt.in)
				defer os.Unsetenv(logLevelEnvName)
				if err != nil {
					t.Fatalf("encountered err: %v", err)
				}

				config, err := NewAppConfig()
				assert.NoError(t, err)
				assert.NotNil(t, config)
				assert.NotNil(t, config.Logger)
				assert.Equal(t, tt.out, config.Logger.Level)
			})
		}
	})
}

func TestNewAppConfigMock(t *testing.T) {
	t.Run("doesn't return an error", func (t *testing.T) {
		config := NewAppConfigMock()
		assert.NotNil(t, config)
	})
}
