package config

import (
	"code.justin.tv/qe/ci_trigger/ci"
	"encoding/json"
	"errors"
	"fmt"
	"github.com/bxcodec/faker"
	"log"
	"strings"
)

const defaultInstanceType = "c4.xlarge"

// Represents a Job that a user wants to trigger
type Job struct {
	ID            string              `json:"job_id"`
	Branch        string              `json:"branch"`
	Platform      string              `json:"platform"`
	Parameters    []ci.BuildParameter `json:"ci_parameters"`
	InstanceType  string              `json:"instanceType"`
	AssignedBuild ci.Build            `faker:"-"`
}

// Returns an array of job objects read from manifest data, which the user defines what jobs they want to run
func NewJobManifest(data []byte, appConfig *App) ([]*Job, error) {
	var jobs []*Job

	err := json.Unmarshal(data, &jobs)
	if err != nil {
		return nil, err
	}

	for _, job := range jobs {
		if job == nil { appConfig.Logger.Warningln("NewJobManifest(): Came across a nil job"); continue }

		if job.InstanceType == "" {
			job.InstanceType = defaultInstanceType
		}
	}

	return jobs, nil
}

// Handles triggering the build on the requested CI System. Currently only supports TeamCity
func (job *Job) TriggerBuild(ciClient ci.Client) (ci.Build, error) {
	// TODO: Support Jenkins
	if strings.ToLower(job.Platform) != "teamcity" {
		return nil, fmt.Errorf("invalid platform. only teamcity implemented. Record: %v", job)
	}

	if ciClient == nil {
		return nil, errors.New("received a nil teamcity client")
	}

	build, err := ciClient.TriggerBuild(job.ID, job.Branch, job.Parameters)
	if err != nil {
		return nil, err
	}

	// assign the created build to the job
	job.AssignedBuild = build
	return build, nil
}

// Returns a Job used for mocks - SHOULD ONLY BE USED IN TESTS
func NewMockJob() *Job {
	job := &Job{}
	err := faker.FakeData(job)
	if err != nil {
		// not returning an error, because this should only be used in Tests. Prevents unnecessary error handling
		log.Fatalf("error creating mock job: %v", err)
	}

	job.Platform = "teamcity" // only teamcity is supported right now
	job.InstanceType = "c4.xlarge"
	return job
}
