package config

import (
	"code.justin.tv/qe/ci_trigger/ci"
	"code.justin.tv/qe/ci_trigger/ci/mocks"
	"code.justin.tv/qe/ci_trigger/teamcity"
	"encoding/json"
	"fmt"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"testing"
)

func TestNewJobManifest(t *testing.T) {
	t.Run("when no jobs", func (t *testing.T) {
		data := "[]"
		jobs, err := NewJobManifest([]byte(data), NewAppConfigMock())
		assert.NoError(t, err)
		assert.Empty(t, jobs)
	})

	t.Run("when 1 job", func (t *testing.T) {
		expectedData := &Job{
			ID:         "1234",
			Branch:     "master",
			Platform:   "teamcity",
			Parameters: []ci.BuildParameter{
				{ Name: "env.BROWSER_NAME", Value: "chrome" },
				{ Name: "env.USE_TESTRAIL", Value: "true" },
			},
			InstanceType: defaultInstanceType,
		}

		// Create a json template out of it
		jsonTemplate, err := json.Marshal(expectedData)
		if err != nil {
			t.Fatalf("error marshaling data: %v", err)
		}

		// Finalize the template data
		data := combineJsonObjects(jsonTemplate)

		// Call the function
		jobs, err := NewJobManifest(data, NewAppConfigMock())
		assert.NoError(t, err)
		assert.Equal(t, 1, len(jobs))
		assert.Equal(t, expectedData, jobs[0])
	})

	t.Run("when multiple", func (t *testing.T) {
		// Create two different job objects and make them into JSON
		expectedData1 := &Job{
			ID:         "1234",
			Branch:     "master",
			Platform:   "teamcity",
			Parameters: []ci.BuildParameter{
				{ Name: "env.BROWSER_NAME", Value: "chrome" },
				{ Name: "env.USE_TESTRAIL", Value: "true" },
			},
			InstanceType: defaultInstanceType,
		}
		jsonTemplate1, err := json.Marshal(expectedData1)
		if err != nil {
			t.Fatalf("error marshaling data: %v", err)
		}
		expectedData2 := &Job{
			ID:         "5678",
			Branch:     "master",
			Platform:   "jenkins",
			Parameters: []ci.BuildParameter{
				{ Name: "env.BROWSER_NAME", Value: "firefox" },
				{ Name: "env.USE_TESTRAIL", Value: "false" },
			},
			InstanceType: defaultInstanceType,
		}
		jsonTemplate2, err := json.Marshal(expectedData2)
		if err != nil {
			t.Fatalf("error marshaling data: %v", err)
		}

		// Finalize the template of the two json objects
		data := combineJsonObjects(jsonTemplate1, jsonTemplate2)

		// Call the function
		jobs, err := NewJobManifest(data, NewAppConfigMock())
		assert.NoError(t, err)
		assert.Equal(t, 2, len(jobs))
		assert.Contains(t, jobs, expectedData1)
		assert.Contains(t, jobs, expectedData2)
	})

	t.Run("when invalid data", func (t *testing.T) {
		data := "not json"
		jobs, err := NewJobManifest([]byte(data), NewAppConfigMock())
		assert.Error(t, err)
		assert.Nil(t, jobs)
	})

	t.Run("sets default instance type if not provided", func (t *testing.T) {
		inputData := &Job{
			ID:         "1234",
			Branch:     "master",
			Platform:   "teamcity",
			Parameters: []ci.BuildParameter{
				{ Name: "env.BROWSER_NAME", Value: "chrome" },
				{ Name: "env.USE_TESTRAIL", Value: "true" },
			},
		}

		// Create a json template out of it
		jsonTemplate, err := json.Marshal(inputData)
		if err != nil {
			t.Fatalf("error marshaling data: %v", err)
		}

		// Finalize the template data
		data := combineJsonObjects(jsonTemplate)

		// Call the function
		jobs, err := NewJobManifest(data, NewAppConfigMock())
		assert.NoError(t, err)
		assert.Equal(t, 1, len(jobs))
		assert.Equal(t, defaultInstanceType, jobs[0].InstanceType, "default instance type should be set")
	})
}

func TestJob_TriggerBuild(t *testing.T) {
	t.Run("when teamcity", func (t *testing.T) {
		job := &Job{
			ID:         "1234",
			Branch:     "master",
			Platform:   "teamcity",
			Parameters: nil,
			InstanceType: defaultInstanceType,
		}
		assert.Nil(t, job.AssignedBuild, "no build should be assigned yet")

		mockClient := &mocks.Client{}
		mockBuild := teamcity.MockBuild("", "")
		mockClient.On("TriggerBuild", mock.Anything, mock.Anything, mock.Anything).Return(mockBuild, nil)

		build, err := job.TriggerBuild(mockClient)
		mockClient.AssertCalled(t, "TriggerBuild", job.ID, job.Branch, job.Parameters)
		assert.NoError(t, err)
		assert.Equal(t, mockBuild, build)
		assert.Equal(t, mockBuild, job.AssignedBuild, "the build should be assigned to the job")
	})

	t.Run("when jenkins", func (t *testing.T) {
		job := &Job{
			ID:         "1234",
			Branch:     "master",
			Platform:   "jenkins",
			Parameters: nil,
			InstanceType: defaultInstanceType,
		}

		mockClient := &mocks.Client{}
		build, err := job.TriggerBuild(mockClient)
		assert.EqualError(t, err, fmt.Sprintf("invalid platform. only teamcity implemented. Record: %v", job))
		assert.Nil(t, build)
	})

	t.Run("when nil client", func (t *testing.T) {
		job := &Job{
			Platform: "teamcity",
		}
		build, err := job.TriggerBuild(nil)
		assert.EqualError(t, err, "received a nil teamcity client")
		assert.Nil(t, build)
	})
}

// Finalizes the template by combining the json objects into a json array
func combineJsonObjects(args ...[]byte) []byte {
	template := "[ " // start out the array

	// Iterate through every json object passed to the function
	for index, val := range args {
		template += string(val)

		// if there are more objects, add a comma
		if index < len(args)-1 {
			template += ",\n"
		}
	}

	// close the array
	template += " ]"
	return []byte(template)
}
