package main

import (
	"code.justin.tv/qe/ci_trigger/ci"
	"code.justin.tv/qe/ci_trigger/config"
	"errors"
	"fmt"
	"log"
	"os"
	"strings"
	"text/tabwriter"
	"time"
)

// Runs a set of jobs, one at a time. If any fail, an error will be returned
// pollDuration is how often to poll TeamCity for the job's status
// timeout is how long to wait for the build to pass, per job
func RunJobs(appConfig *config.App, jobs []*config.Job, client ci.Client, targetInstanceType string,
	pollDuration time.Duration, timeout time.Duration) error {
	if client == nil {
		return errors.New("received a nil teamcity client")
	}

	// Trigger each job and wait for it to complete
	for _, job := range jobs {
		if job == nil { log.Println("RunJobs(): [WARN] Came across a nil job"); continue }

		if strings.ToLower(job.InstanceType) != strings.ToLower(targetInstanceType) {
			appConfig.Logger.Infof("skipping job: '%s' as the target instance type '%s' does not match the job type: '%s'",
				job.ID, targetInstanceType, job.InstanceType)
			continue
		}

		err := triggerJobAndWaitToComplete(appConfig, job, client, pollDuration, timeout)
		if err != nil {
			return err
		}
	}

	return nil
}

// Triggers a specific job
// Returns an error if the build failed, or it timed out waiting
// pollDuration is how often to poll TeamCity for the job's status
// timeout is how long to wait for the build to pass, per job
func triggerJobAndWaitToComplete(appConfig *config.App, job *config.Job, client ci.Client,
	  pollDuration time.Duration, timeout time.Duration) error {
	if job == nil { return errors.New("cannot trigger a nil job") }

	appConfig.Logger.Infof("Triggering Job: %v", job)
	triggeredBuild, err := job.TriggerBuild(client)
	if err != nil {
		return err
	}

	// Make sure a valid ID was returned
	if triggeredBuild.ID() == 0 {
		return fmt.Errorf("the triggered build's ID was not set. Job: %v Triggered Build: %v", job, triggeredBuild)
	}

	appConfig.Logger.Infof("Build triggered: %v", triggeredBuild)

	// Wait for build to complete within a specified timeout
	deadline := time.Now().Add(timeout)
	appConfig.Logger.Infof("Waiting for Build #%d to complete. Will timeout by: %s", triggeredBuild.ID(), deadline)
	for time.Now().Before(deadline) {
		time.Sleep(pollDuration) // at the beginning to give some time for the build queue to start

		// Get the build information
		build, err := client.GetBuild(triggeredBuild.ID())
		if err != nil {
			appConfig.Logger.Warnf("Received error while fetching build: %v", err)
			continue // try again...
		}
		appConfig.Logger.Debugf("Retrieved build from GetBuild: %+v", build)

		// assign the updated build to the job
		job.AssignedBuild = build

		if build.Complete() {
			appConfig.Logger.Infof("Build completed: %s: %s", build.JobName(), build.State())

			// return no error if it passed. If it failed, return an error
			if build.Passed() {
				return nil
			} else {
				return fmt.Errorf("build did not succeed. build: %v", build)
			}
		} else { // build was not complete, so retry
			appConfig.Logger.Debugf("Build not completed. Will retry...")
		}
	}

	// Hits here if the timout was reached
	return fmt.Errorf("timeout reached for build: %v", triggeredBuild)
}

// PrintResults will write the results of the job to the screen, with formatting
func PrintResults(jobs []*config.Job) {
	dashes := "------------------"
	fmt.Printf("\n\n%s\nResults:\n\n", dashes) // opener header

	// Create a TabWriter to use for printing out results in nice formatting
	writer := tabwriter.NewWriter(os.Stdout, 0, 8, 2, '\t', tabwriter.AlignRight)

	// Headers
	_, _ = fmt.Fprintln(writer, "Job ID\tResult\tURL")

	// Loop through each job and report their results
	for _, job := range jobs {
		if job == nil { log.Println("PrintResults(): [WARN] Came across a nil job"); continue }

		if job.AssignedBuild == nil { // incase a build wasn't assigned, print N/A for the results
			_, _ = fmt.Fprintf(writer, "%s\tN/A\tN/A\n", job.ID)
			continue
		}
		_, _ = fmt.Fprintf(writer, "%s\t%s\t%s\n", job.ID, job.AssignedBuild.Status(), job.AssignedBuild.WebURL())
	}

	// Write the results to stdout
	err := writer.Flush()
	if err != nil {
		log.Printf("error writing results: %v", err)
	}

	fmt.Printf("\n%s\n", dashes) // closer header
}
