package main

import (
	"code.justin.tv/qe/ci_trigger/ci/mocks"
	"code.justin.tv/qe/ci_trigger/config"
	"code.justin.tv/qe/ci_trigger/teamcity"
	"errors"
	"fmt"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"testing"
	"time"
)

func TestRunJobs(t *testing.T) {
	targetInstanceType := "c4.xlarge"

	t.Run("returns error when teamcity client nil", func (t *testing.T) {
		jobs := make([]*config.Job, 0)
		err := RunJobs(config.NewAppConfigMock(), jobs, nil, targetInstanceType, time.Nanosecond, time.Millisecond)
		assert.EqualError(t, err, "received a nil teamcity client")
	})

	t.Run("triggers build for each job and returns no error if all pass", func (t *testing.T) {
		jobs := []*config.Job{
			config.NewMockJob(),
			config.NewMockJob(),
		}

		mockClient := mocks.Client{}

		mockTCBuild1 := teamcity.MockBuild("queued", "pending")
		mockTCBuild2 := teamcity.MockBuild("queued", "pending")

		// Copy the build into a new variable, and set the state and status to complete
		// The GetBuild function can then return this completed build
		mockCompleteBuild1 := *mockTCBuild1 // copy
		mockCompleteBuild1.SetState(teamcity.CompleteLabel)
		mockCompleteBuild1.SetStatus(teamcity.PassedLabel)

		// Copy the build into a new variable, and set the state and status to complete
		// The GetBuild function can then return this completed build
		mockCompleteBuild2 := *mockTCBuild2 //copy
		mockCompleteBuild2.SetState(teamcity.CompleteLabel)
		mockCompleteBuild2.SetStatus(teamcity.PassedLabel)

		mockClient.On("TriggerBuild", jobs[0].ID, jobs[0].Branch, jobs[0].Parameters).Return(mockTCBuild1, nil)
		mockClient.On("TriggerBuild", jobs[1].ID, jobs[1].Branch, jobs[1].Parameters).Return(mockTCBuild2, nil)

		mockClient.On("GetBuild", mockTCBuild1.ID()).Return(&mockCompleteBuild1, nil)
		mockClient.On("GetBuild", mockTCBuild2.ID()).Return(&mockCompleteBuild2, nil)

		err := RunJobs(config.NewAppConfigMock(), jobs, &mockClient, targetInstanceType, time.Nanosecond, time.Millisecond)
		assert.NoError(t, err)
		mockClient.AssertCalled(t, "TriggerBuild", jobs[0].ID, jobs[0].Branch, jobs[0].Parameters)
		mockClient.AssertCalled(t, "TriggerBuild", jobs[1].ID, jobs[1].Branch, jobs[1].Parameters)
		mockClient.AssertCalled(t, "GetBuild", mockTCBuild1.ID())
		mockClient.AssertCalled(t, "GetBuild", mockTCBuild2.ID())
	})

	t.Run("returns error if failed and does not run the second", func (t *testing.T) {
		jobs := []*config.Job{
			config.NewMockJob(),
			config.NewMockJob(),
		}

		mockClient := mocks.Client{}

		mockTCBuild1 := teamcity.MockBuild("queued", "pending")
		mockTCBuild2 := teamcity.MockBuild("queued", "pending")

		mockCompleteBuild1 := *mockTCBuild1 // Copy the build into a new variable, and set the state and status to complete
		mockCompleteBuild1.SetState(teamcity.CompleteLabel)
		mockCompleteBuild1.SetStatus("failed") // FAIL THE BUILD!

		mockCompleteBuild2 := *mockTCBuild2 // Copy the build into a new variable, and set the state and status to complete
		mockCompleteBuild2.SetState(teamcity.CompleteLabel)
		mockCompleteBuild2.SetStatus(teamcity.PassedLabel)

		mockClient.On("TriggerBuild", jobs[0].ID, jobs[0].Branch, jobs[0].Parameters).Return(mockTCBuild1, nil)
		mockClient.On("TriggerBuild", jobs[1].ID, jobs[1].Branch, jobs[1].Parameters).Return(mockTCBuild2, nil)

		mockClient.On("GetBuild", mockTCBuild1.ID()).Return(&mockCompleteBuild1, nil)
		mockClient.On("GetBuild", mockTCBuild2.ID()).Return(&mockCompleteBuild2, nil)

		err := RunJobs(config.NewAppConfigMock(), jobs, &mockClient, targetInstanceType, time.Nanosecond, time.Millisecond)
		assert.EqualError(t, err, fmt.Sprintf("build did not succeed. build: %v", &mockCompleteBuild1))
		mockClient.AssertCalled(t, "TriggerBuild", jobs[0].ID, jobs[0].Branch, jobs[0].Parameters)
		mockClient.AssertNotCalled(t, "TriggerBuild", jobs[1].ID, jobs[1].Branch, jobs[1].Parameters, "second build should not have been called as the first failed")
		mockClient.AssertCalled(t, "GetBuild", mockTCBuild1.ID())
		mockClient.AssertNotCalled(t, "GetBuild", mockTCBuild2.ID(), "second build should not have been called as the first failed")
	})

	t.Run("returns error if timeout exceeded", func (t *testing.T) {
		jobs := []*config.Job{
			config.NewMockJob(),
			config.NewMockJob(),
		}

		mockClient := mocks.Client{}

		mockTCBuild1 := teamcity.MockBuild("queued", "pending")
		mockTCBuild2 := teamcity.MockBuild("queued", "pending")

		mockCompleteBuild1 := *mockTCBuild1 // copy
		mockCompleteBuild2 := *mockTCBuild2 //copy

		mockClient.On("TriggerBuild", jobs[0].ID, jobs[0].Branch, jobs[0].Parameters).Return(mockTCBuild1, nil)
		mockClient.On("TriggerBuild", jobs[1].ID, jobs[1].Branch, jobs[1].Parameters).Return(mockTCBuild2, nil)

		mockClient.On("GetBuild", mockTCBuild1.ID()).Return(&mockCompleteBuild1, nil)
		mockClient.On("GetBuild", mockTCBuild2.ID()).Return(&mockCompleteBuild2, nil)

		err := RunJobs(config.NewAppConfigMock(), jobs, &mockClient, targetInstanceType, time.Nanosecond, time.Millisecond)
		assert.EqualError(t, err, fmt.Sprintf("timeout reached for build: %v", &mockCompleteBuild1))
		mockClient.AssertCalled(t, "TriggerBuild", jobs[0].ID, jobs[0].Branch, jobs[0].Parameters)
		mockClient.AssertNotCalled(t, "TriggerBuild", jobs[1].ID, jobs[1].Branch, jobs[1].Parameters, "second build should not have been called as the first failed")
		mockClient.AssertCalled(t, "GetBuild", mockTCBuild1.ID())
		mockClient.AssertNotCalled(t, "GetBuild", mockTCBuild2.ID(), "second build should not have been called as the first failed")
	})

	t.Run("returns error if client fails to trigger build", func (t *testing.T) {
		jobs := []*config.Job{config.NewMockJob(),}

		mockClient := mocks.Client{}
		mockError := errors.New("unexpected error")

		mockClient.On("TriggerBuild", mock.Anything, mock.Anything, mock.Anything).Return(nil, mockError)
		err := RunJobs(config.NewAppConfigMock(), jobs, &mockClient, targetInstanceType, time.Nanosecond, time.Millisecond)
		assert.Equal(t, mockError, err)
	})

	t.Run("keeps trying if client returns GetBuild failures", func (t *testing.T) {
		jobs := []*config.Job{config.NewMockJob(),}

		mockClient := mocks.Client{}
		mockError := errors.New("unexpected error")

		mockTCBuild1 := teamcity.MockBuild("queued", "pending")
		// No need for mockTCBuild2 - it should never try the second
		mockClient.On("TriggerBuild", jobs[0].ID, jobs[0].Branch, jobs[0].Parameters).Return(mockTCBuild1, nil)

		mockClient.On("GetBuild", mock.Anything).Return(nil, mockError)
		err := RunJobs(config.NewAppConfigMock(), jobs, &mockClient, targetInstanceType, time.Nanosecond, time.Millisecond)

		// It should be a timeout error
		assert.EqualError(t, err, fmt.Sprintf("timeout reached for build: %v", mockTCBuild1))
	})

	t.Run("returns an error if triggered build returns an invalid id", func (t *testing.T) {
		jobs := []*config.Job{config.NewMockJob(),}

		mockClient := mocks.Client{}
		mockTCBuild1 := &teamcity.Build{} // create a non-initialized Build to test the zero-type of ID
		mockClient.On("TriggerBuild", jobs[0].ID, jobs[0].Branch, jobs[0].Parameters).Return(mockTCBuild1, nil)

		err := RunJobs(config.NewAppConfigMock(), jobs, &mockClient, targetInstanceType, time.Nanosecond, time.Millisecond)

		jobs[0].AssignedBuild = mockTCBuild1 // the error will have a job assigned to it. Attach that to the job so the error is accurate
		assert.EqualError(t, err, fmt.Sprintf("the triggered build's ID was not set. Job: %v Triggered Build: %v", jobs[0], mockTCBuild1))
	})

	t.Run("only runs jobs matching the target instance type", func (t *testing.T) {
		const (
			regInstanceType  = "c4.xlarge"
			perfInstanceType = "c4.2xlarge"
		)

		mockClient := mocks.Client{}

		// Create two jobs that uses the Regression Test instance type
		mockJobReg1 := config.NewMockJob()
		mockJobReg1.InstanceType = regInstanceType

		mockJobReg2 := config.NewMockJob()
		mockJobReg2.InstanceType = regInstanceType

		// Create one job that uses the Performance Test instance type
		mockJobPerf1 := config.NewMockJob()
		mockJobPerf1.InstanceType = perfInstanceType

		// Load these into an array
		jobs := []*config.Job{
			mockJobReg1,
			mockJobPerf1, // purposely putting performance in the middle
			mockJobReg2,
		}

		// On Trigger/Get, return a mock build for the first regression job
		mockTCBuildReg1 := teamcity.MockBuild(teamcity.PassedLabel, teamcity.CompleteLabel)
		mockClient.On("TriggerBuild", mockJobReg1.ID, mockJobReg1.Branch, mockJobReg1.Parameters).Return(mockTCBuildReg1, nil)
		mockClient.On("GetBuild", mockTCBuildReg1.ID()).Return(mockTCBuildReg1, nil)

		// On Trigger/Get, return a mock build for the second regression job
		mockTCBuildReg2 :=  teamcity.MockBuild(teamcity.PassedLabel, teamcity.CompleteLabel)
		mockClient.On("TriggerBuild", mockJobReg2.ID, mockJobReg2.Branch, mockJobReg2.Parameters).Return(mockTCBuildReg2, nil)
		mockClient.On("GetBuild", mockTCBuildReg2.ID()).Return(mockTCBuildReg2, nil)

		// On Trigger/Get, return a mock build for the first performance job - this shouldn't be called (asserted below) but just incase...
		mockTCBuildPerf1 :=  teamcity.MockBuild(teamcity.PassedLabel, teamcity.CompleteLabel)
		mockClient.On("TriggerBuild", mockJobPerf1.ID, mockJobPerf1.Branch, mockJobPerf1.Parameters).Return(mockTCBuildPerf1, nil)
		mockClient.On("GetBuild", mockTCBuildPerf1.ID()).Return(mockTCBuildPerf1, nil)

		// Make the actual call
		err := RunJobs(config.NewAppConfigMock(), jobs, &mockClient, targetInstanceType, time.Nanosecond, time.Millisecond)
		assert.NoError(t, err)

		// Assert that the proper regression jobs were triggered
		mockClient.AssertCalled(t, "TriggerBuild", mockJobReg1.ID, mockJobReg1.Branch, mockJobReg1.Parameters)
		mockClient.AssertCalled(t, "TriggerBuild", mockJobReg2.ID, mockJobReg2.Branch, mockJobReg2.Parameters)

		// Assert that the performance job was not triggered
		mockClient.AssertNotCalled(t, "TriggerBuild", mockJobPerf1.ID, mockJobPerf1.Branch, mockJobPerf1.Parameters)
	})

	t.Run("assigns the build to the job", func (t *testing.T) {
		mockClient := mocks.Client{}

		jobs := []*config.Job{config.NewMockJob(),}
		assert.Nil(t, jobs[0].AssignedBuild)

		// On Trigger/Get, return a mock build for the first regression job
		mockTCBuild := teamcity.MockBuild(teamcity.PassedLabel, teamcity.CompleteLabel)
		mockClient.On("TriggerBuild", jobs[0].ID, jobs[0].Branch, jobs[0].Parameters).Return(mockTCBuild, nil)
		mockClient.On("GetBuild", mockTCBuild.ID()).Return(mockTCBuild, nil)

		err := RunJobs(config.NewAppConfigMock(), jobs, &mockClient, targetInstanceType, time.Nanosecond, time.Millisecond)
		assert.NoError(t, err)
		assert.Equal(t, mockTCBuild, jobs[0].AssignedBuild)
	})
}
