package teamcity

import (
	"code.justin.tv/qe/ci_trigger/ci"
	"encoding/json"
	"github.com/bxcodec/faker"
	"log"
	"strings"
)

const (
	PassedLabel = "success" // PassedLabel is the label TeamCity applies if a build passed (successful)
	CompleteLabel = "finished" // CompleteLabel is the label TeamCity applies if a build is complete (finished)
)

// Holds the JSON Data for a build.
// This way the json package has access to the exported fields, without the interface duplicating methods
type jsonData struct {
	ID            int    `json:"id"`
	BuildTypeId   string `json:"buildTypeId"`
	Number        string `json:"number"`
	Status        string `json:"status"`
	State         string `json:"state"`
	BranchName    string `json:"branchName"`
	DefaultBranch bool   `json:"defaultBranch"`
	Href          string `json:"href"`
	WebURL        string `json:"webUrl"`
	StatusText    string `json:"statusText"`
}

// Represents a TeamCity Build
type Build struct {
	jsonData
}

// Data to send to TeamCity to trigger a build
type BuildInput struct {
	BranchName  string               `json:"branchName"`
	BuildType   buildType            `json:"buildType"`
	Properties  BuildInputProperties `json:"properties"`
}

// BuildInputProperties are the properties value to pass to TeamCity.
// Within it, an array of parameter/properties are held
type BuildInputProperties struct {
	Property []ci.BuildParameter `json:"property"`
}

// Stores the Build ID
type buildType struct {
	ID string `json:"id"`
}

// Unmarshal's the JSON Data
func (b *Build) UnmarshalJSON(data []byte) error {
	return json.Unmarshal(data, &b.jsonData)
}

// Marshal's Json Data
func (b *Build) MarshalJSON() ([]byte, error) {
	return json.Marshal(&b.jsonData)
}

// Returns a Build's ID
func (b *Build) ID() int {
	return b.jsonData.ID
}

// Returns a Build's Name
func (b *Build) JobName() string {
	return b.jsonData.BuildTypeId
}

// Returns a Build's Status
func (b *Build) Status() string {
	return b.jsonData.Status
}

func (b *Build) SetStatus(status string) {
	b.jsonData.Status = status
}

// Returns a Build's State
func (b *Build) State() string {
	return b.jsonData.State
}

func (b *Build) SetState(state string) {
	b.jsonData.State = state
}

// Returns a Build's Branch Name
func (b *Build) BranchName() string {
	return b.jsonData.BranchName
}

// Returns whether a build is complete
func (b *Build) Complete() bool {
	return strings.ToLower(b.jsonData.State) == CompleteLabel
}

// Returns whether a build passed (true = passed, false = failed)
func (b *Build) Passed() bool {
	return strings.ToLower(b.jsonData.Status) == PassedLabel
}

// Returns the web url of the job
func (b *Build) WebURL() string {
	return b.jsonData.WebURL
}

// Returns a build used for mocks - SHOULD ONLY BE USED IN TESTS
// status: supply "" for default faked data
// state:  supply "" for default faked data
func MockBuild(status, state string) *Build {
	data := jsonData{}
	err := faker.FakeData(&data)
	if err != nil {
		// not returning an error, because this should only be used in Tests. Prevents unnecessary error handling
		log.Fatalf("error creating mock build: %v", err)
	}
	if status != "" {
		data.Status = status
	}
	if state != "" {
		data.State = state
	}
	return &Build{data}
}
