package teamcity

import (
	"bytes"
	"encoding/json"
	"github.com/stretchr/testify/assert"
	"testing"
	"text/template"
)

func TestBuild_UnmarshalJSON(t *testing.T) {
	t.Run("given a valid json blob, unmarshal's json", func (t *testing.T) {
		b := Build{}

		// Refer to a template of JSON Data for testing Unmarshal
		// Generate a struct with that data, load the template and execute the template
		expectedJSONData := jsonData{
			ID:            1234,
			BuildTypeId:   "myTestTypeID",
			Number:        "master-1234",
			Status:        "success",
			State:         "complete",
			BranchName:    "master",
			DefaultBranch: true,
			Href:          "/httpAuth/app/rest/latest/builds/id:1234",
			WebURL:        "https://teamcity.xarth.tv/viewLog.html?buildId=1234&buildTypeId=myTestTypeID",
			StatusText:    "Tests passed: 330, ignored: 60",
		}

		temp, err := template.ParseFiles("./test_data/build_valid_resp.json")
		if err != nil {
			t.Fatalf("encountered error reading file: %v", err)
		}

		// Execute the template and write it into a buffer
		buf := new(bytes.Buffer)
		err = temp.Execute(buf, expectedJSONData)
		if err != nil {
			t.Fatalf("encountered error executing template: %v", err)
		}

		// Execute the Unmarshal
		err = b.UnmarshalJSON(buf.Bytes())
		assert.NoError(t, err)

		// Make sure the fields equal
		assert.Equal(t, expectedJSONData, b.jsonData)
	})

	t.Run("give a non json blob, returns an error", func (t *testing.T) {
		b := Build{}
		err := b.UnmarshalJSON([]byte("server error"))
		assert.Error(t, err)
	})
}

func TestBuild_MarshalJSON(t *testing.T) {
	t.Run("given a valid build, exports to valid json", func (t *testing.T) {
		build := Build{jsonData{
			ID:            1234,
			BuildTypeId:   "myTestTypeID",
			Number:        "master-1234",
			Status:        "success",
			State:         "complete",
			BranchName:    "master",
			DefaultBranch: true,
			Href:          "/my/path",
			WebURL:        "http://my.url",
			StatusText:    "Tests Running",
		}}

		data, err := build.MarshalJSON()
		assert.NoError(t, err)

		assert.True(t, json.Valid(data))

		// now make sure it matches
		actualBuildFromJson := Build{}
		err = json.Unmarshal(data, &actualBuildFromJson.jsonData)
		assert.NoError(t, err)

		assert.Equal(t, build, actualBuildFromJson)
	})
}

func TestBuild_ID(t *testing.T) {
	t.Run("returns ID when valid", func (t *testing.T) {
		build := Build{jsonData{ID: 1234}}
		assert.Equal(t, build.jsonData.ID, build.ID())
	})
}

func TestBuild_JobName(t *testing.T) {
	t.Run("returns BuildTypeId when valid", func (t *testing.T) {
		build := Build{jsonData{BuildTypeId: "myjob"}}
		assert.Equal(t, build.jsonData.BuildTypeId, build.JobName())
	})
}

func TestBuild_Status(t *testing.T) {
	t.Run("returns Status when valid", func (t *testing.T) {
		build := Build{jsonData{Status: "success"}}
		assert.Equal(t, build.jsonData.Status, build.Status())
	})
}

func TestBuild_SetStatus(t *testing.T) {
	originalStatus := "first"
	newStatus := "second"
	build := Build{jsonData{Status: originalStatus}}
	assert.Equal(t, originalStatus, build.jsonData.Status)
	build.SetStatus(newStatus)
	assert.Equal(t, newStatus, build.jsonData.Status)
}

func TestBuild_State(t *testing.T) {
	t.Run("returns State when valid", func (t *testing.T) {
		build := Build{jsonData{State: "complete"}}
		assert.Equal(t, build.jsonData.State, build.State())
	})
}

func TestBuild_SetState(t *testing.T) {
	originalState := "first"
	newState := "second"
	build := Build{jsonData{State: originalState}}
	assert.Equal(t, originalState, build.jsonData.State)
	build.SetState(newState)
	assert.Equal(t, newState, build.jsonData.State)
}

func TestBuild_BranchName(t *testing.T) {
	t.Run("returns State when valid", func (t *testing.T) {
		build := Build{jsonData{BranchName: "master"}}
		assert.Equal(t, build.jsonData.BranchName, build.BranchName())
	})
}

func TestBuild_Complete(t *testing.T) {
	var completeTests = []struct {
		in  string
		out bool
	}{
		{"finished", true},
		{"FINISHED", true},
		{"running", false},
		{"", false},
	}
	for _, tt := range completeTests {
		t.Run(tt.in, func(t *testing.T) {
			build := Build{jsonData{State: tt.in}}
			assert.Equal(t, tt.out, build.Complete())
		})
	}
}

func TestBuild_Passed(t *testing.T) {
	var passedTests = []struct {
		in  string
		out bool
	}{
		{"success", true},
		{"SUCCESS", true},
		{"failed", false},
		{"", false},
	}
	for _, tt := range passedTests {
		t.Run(tt.in, func(t *testing.T) {
			build := Build{jsonData{Status: tt.in}}
			assert.Equal(t, tt.out, build.Passed())
		})
	}
}

func TestBuild_WebURL(t *testing.T) {
	t.Run("returns Web URL when valid", func (t *testing.T) {
		build := Build{jsonData{WebURL: "http://www.weburl.com"}}
		assert.Equal(t, build.jsonData.WebURL, build.WebURL())
	})
}

func TestMockBuild(t *testing.T) {
	t.Run("returns a build", func(t *testing.T) {
		build := MockBuild("", "")
		assert.NotNil(t, build)
		assert.NotEmpty(t, build)
		assert.NotEmpty(t, build.ID())
	})

	t.Run("sets status", func(t *testing.T) {
		expectedStatus := "mystatus"
		build := MockBuild(expectedStatus, "")
		assert.NotNil(t, build)
		assert.NotEmpty(t, build)
		assert.NotEmpty(t, build.ID())
		assert.Equal(t, expectedStatus, build.jsonData.Status)
	})

	t.Run("sets state", func(t *testing.T) {
		expectedState := "mystate"
		build := MockBuild("", expectedState)
		assert.NotNil(t, build)
		assert.NotEmpty(t, build)
		assert.NotEmpty(t, build.ID())
		assert.Equal(t, expectedState, build.jsonData.State)
	})
}

