package teamcity

import (
	"bytes"
	"code.justin.tv/qe/ci_trigger/ci"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"time"
)

// A client for TeamCity
type Client struct {
	httpClient *http.Client
	scheme     string
	host       string
	username   string
	password   string
	version    string
	config     *Config
}

// Creates a new TeamCity Client
// host: The teamcity host, such as teamcity.xarth.tv
// config: A TeamCity Config with a username & password. You can create one via teamcity.NewConfig()
// version: The api version to use (such as "latest")
func New(host string, config *Config, version string) (*Client, error) {
	if config == nil  {
		return nil, errors.New("teamcity config was nil")
	}

	return &Client{
		httpClient: &http.Client{
		  Timeout: time.Second * 30, // teamcity responses can be slow...
		},
		scheme:     "https",
		host:       host,
		username:   config.Username,
		password:   config.Password,
		version:    version,
		config:     config,
	}, nil
}

// Returns the http client used
func (c *Client) HTTPClient() *http.Client {
	return c.httpClient
}

// Returns the host
func (c *Client) Host() string {
	return c.host
}

// Returns the fully qualified root url for the API
func (c *Client) RootAPIURL() string {
	return fmt.Sprintf("%s://%s/httpAuth/app/rest/%s", c.scheme, c.host, c.version)
}

// Returns a build by the specified Build ID
func (c *Client) GetBuild(id int) (ci.Build, error) {
	path := fmt.Sprintf("/builds/id:%d", id)

	// Pass the request to the client
	resp, err := c.makeRequest(http.MethodGet, path, nil)
	if err != nil {
		return nil, err
	}

	// Open the body
	defer resp.Body.Close()
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}

	// Make sure the request was successful
	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("expected status code %d. Got: %d. Body: %s", http.StatusOK, resp.StatusCode, string(body))
	}

	// Fill the build object
	build := &Build{}
	err = build.UnmarshalJSON(body)
	if err != nil {
		return nil, fmt.Errorf("error unmarshaling json: %v", err)
	}

	return build, nil
}

// Triggers a TeamCity Build
func (c *Client) TriggerBuild(jobId string, branchName string, properties []ci.BuildParameter) (ci.Build, error) {
	path := "/buildQueue"

	// Store the requested properties into a struct
	buildProperties := BuildInputProperties{ properties }

	// Generate input to pass to TeamCity with the build details
	input := &BuildInput{
		BranchName: branchName,
		BuildType: buildType{ ID: jobId },
		Properties: buildProperties,
	}

	// Convert to JSON
	output, err := json.Marshal(&input)
	if err != nil {
		return nil, fmt.Errorf("error marshaling json: %v", err)
	}

	// Make the request to the client
	resp, err := c.makeRequest(http.MethodPost, path, bytes.NewBuffer(output))
	if err != nil {
		return nil, err
	}

	// Open the body
	defer resp.Body.Close()
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}

	// Ensure it was successful
	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("expected status code %d. Got: %d. Body: %s", http.StatusOK, resp.StatusCode, string(body))
	}

	// Fill the build object from the details
	build := &Build{}
	err = build.UnmarshalJSON(body)
	if err != nil {
		return nil, fmt.Errorf("error unmarshaling json: %v", err)
	}

	return build, nil
}

// Makes a request to the TeamCity server
// Handles redundant logic like basic auth, fetching the root api url, etc
// Pass a method, the relative path, and the body
func (c *Client) makeRequest(method string, path string, body io.Reader) (*http.Response, error) {
	fullURL := fmt.Sprintf("%s%s", c.RootAPIURL(), path) // build the full url

	// Create the request
	request, err := http.NewRequest(method, fullURL, body)
	if err != nil {
		return nil, err
	}
	request.SetBasicAuth(c.username, c.password)
	request.Header.Set("Content-Type", "application/json")
	request.Header.Set("Accept", "application/json")

	// Make the request (with some nice logging)
	c.config.Logger.Debugf("-> %s %s", method, fullURL)
	resp, err := c.HTTPClient().Do(request)
	if err != nil {
		return nil, err
	}
	c.config.Logger.Debugf("[%d] <- %s %s", resp.StatusCode, method, fullURL)
	return resp, nil
}
