package teamcity

import (
	"bytes"
	"code.justin.tv/qe/ci_trigger/config"
	"encoding/json"
	"fmt"
	"github.com/stretchr/testify/assert"
	"io/ioutil"
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"
	"text/template"
	"time"
)

func TestNew(t *testing.T) {
	t.Run("returns a new client", func (t *testing.T) {
		host := "testHost"
		user := "testUser"
		pass := "testPass"
		version := "testVersion"

		// Set up the config
		tcConfig := NewConfigMock()
		tcConfig.Username = user
		tcConfig.Password = pass

		// Initialize the client
		actualClient, err := New(host, tcConfig, version)
		assert.NoError(t, err)
		assert.Equal(t, host, actualClient.host)
		assert.Equal(t, user, actualClient.username)
		assert.Equal(t, pass, actualClient.password)
		assert.Equal(t, version, actualClient.version)
		assert.Equal(t, tcConfig, actualClient.config)
		assert.Equal(t, "https", actualClient.scheme, "should default to https")
		assert.NotNil(t, actualClient.httpClient)
	})

	t.Run("returns error if no teamcity config is provided", func (t *testing.T) {
		client, err := New("testHost", nil, "latest")
		assert.EqualError(t, err, "teamcity config was nil")
		assert.Nil(t, client)
	})
}

func TestClient_HTTPClient(t *testing.T) {
	httpClient := &http.Client{
		Timeout: time.Second * 1,
	}
	client := Client{httpClient: httpClient}
	assert.Equal(t, httpClient, client.HTTPClient())
}

func TestClient_Host(t *testing.T) {
	host := "my.host"
	client := Client{host: host}
	assert.Equal(t, host, client.Host())
}

func TestClient_RootAPIURL(t *testing.T) {
	scheme := "https"
	host := "my.host"
	version := "latest"

	client := Client{
		scheme: scheme,
		host: host,
		version: version,
	}

	expectedRootURL := fmt.Sprintf("%s://%s/httpAuth/app/rest/%s", scheme, host, version)
	assert.Equal(t, expectedRootURL, client.RootAPIURL())
}

func TestClient_GetBuild(t *testing.T) {
	// Set up function scoped variables so that through this test function we can change what the test server responds
	returnCode := http.StatusOK
	var returnData = new(bytes.Buffer)

	tcServer := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if returnCode != http.StatusOK {
			http.Error(w, "unexpected error", returnCode)
		} else {
			w.Header().Set("Content-Type", "application/json")
			_, err := fmt.Fprint(w, returnData)
			if err != nil {
				t.Logf("error when writing data: %v", err)
			}
		}
	}))
	defer tcServer.Close()

	tcServerURL, err := url.Parse(tcServer.URL)
	if err != nil {
		t.Fatalf("error parsing server url: %v", err)
	}

	client, err := New(tcServerURL.Host, NewConfigMock(), "latest")
	if err != nil {
		t.Fatalf("error creating teamcity client: %v", err)
	}
	client.scheme = "http" // test server is in http...

	t.Run("returns build data", func (t *testing.T) {
		expectedBuild := MockBuild("", "")
		temp, err := template.ParseFiles("./test_data/build_valid_resp.json")
		if err != nil {
			t.Fatalf("encountered error reading file: %v", err)
		}

		returnData = new(bytes.Buffer)
		err = temp.Execute(returnData, expectedBuild)
		if err != nil {
			t.Fatalf("encountered error executing template: %v", err)
		}

		actualBuild, err := client.GetBuild(expectedBuild.ID())
		assert.NoError(t, err)
		assert.Equal(t, expectedBuild, actualBuild)
	})

	t.Run("returns error if server does not return 200", func (t *testing.T) {
		returnData = new(bytes.Buffer)
		returnCode = 500
		actualBuild, err := client.GetBuild(MockBuild("", "").ID())
		assert.EqualError(t, err, fmt.Sprintf("expected status code 200. Got: %d. Body: unexpected error\n", returnCode))
		assert.Nil(t, actualBuild)
	})

	t.Run("returns error if invalid json", func (t *testing.T) {
		returnData = new(bytes.Buffer)
		_, err = fmt.Fprint(returnData, "not json")
		if err != nil {
			t.Logf("error writing to buffer: %v", err)
		}

		returnCode = http.StatusOK
		actualBuild, err := client.GetBuild(MockBuild("", "").ID())
		assert.EqualError(t, err, "error unmarshaling json: invalid character 'o' in literal null (expecting 'u')")
		assert.Nil(t, actualBuild)
	})
}

func TestClient_TriggerBuild(t *testing.T) {
	// Set up function scoped variables so that through this test function we can change what the test server responds
	returnCode := http.StatusOK
	var returnData = new(bytes.Buffer)
	var receivedData []byte // to ensure the server received the correct data

	tcServer := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		var err error

		defer r.Body.Close()
		receivedData, err = ioutil.ReadAll(r.Body)
		if err != nil {
			t.Logf("error reading body in http server")
		}

		if returnCode != http.StatusOK {
			http.Error(w, "unexpected error", returnCode)
		} else {
			w.Header().Set("Content-Type", "application/json")
			_, err := fmt.Fprint(w, returnData)
			if err != nil {
				t.Logf("error when writing data: %v", err)
			}
		}
	}))
	defer tcServer.Close()

	tcServerURL, err := url.Parse(tcServer.URL)
	if err != nil {
		t.Fatalf("error parsing server url: %v", err)
	}

	client, err := New(tcServerURL.Host, NewConfigMock(), "latest")
	if err != nil {
		t.Fatalf("error creating teamcity client: %v", err)
	}
	client.scheme = "http"

	t.Run("triggers a build", func (t *testing.T) {
		// Generate a mock job to trigger to the server
		jobToTrigger := config.NewMockJob()

		// Create a mock build to return from the server, using data from the mock job
		expectedBuild := &Build{jsonData{
			ID:            1234,
			BuildTypeId:   jobToTrigger.ID,
			Number:        "1234",
			Status:        "queued",
			State:         "queued",
			BranchName:    jobToTrigger.Branch,
			DefaultBranch: true,
			Href:          "/href",
			WebURL:        "www.url.com/href",
			StatusText:    "queued",
		}}

		// Generate the JSON Template for the server to return
		temp, err := template.ParseFiles("./test_data/build_valid_resp.json")
		if err != nil {
			t.Fatalf("encountered error reading file: %v", err)
		}

		// Set the server's return data to the template
		returnData = new(bytes.Buffer)
		err = temp.Execute(returnData, expectedBuild)
		if err != nil {
			t.Fatalf("encountered error executing template: %v", err)
		}

		// Make the request
		actualBuild, err := client.TriggerBuild(jobToTrigger.ID, jobToTrigger.Branch, jobToTrigger.Parameters)
		assert.NoError(t, err)
		assert.Equal(t, expectedBuild, actualBuild)

		// assert the server received the correct things
		serverReceivedBuildInput := &BuildInput{}
		err = json.Unmarshal(receivedData, serverReceivedBuildInput)
		if err != nil {
			t.Logf("error unmarshalling data")
		}

		assert.Equal(t, jobToTrigger.ID, serverReceivedBuildInput.BuildType.ID)
		assert.Equal(t, jobToTrigger.Branch, serverReceivedBuildInput.BranchName)
		assert.Equal(t, jobToTrigger.Parameters, serverReceivedBuildInput.Properties.Property)
	})

	t.Run("returns error if server does not return 200", func (t *testing.T) {
		returnData = new(bytes.Buffer)
		returnCode = 500
		actualBuild, err := client.TriggerBuild("job", "master", nil)
		assert.EqualError(t, err, fmt.Sprintf("expected status code 200. Got: %d. Body: unexpected error\n", returnCode))
		assert.Nil(t, actualBuild)
	})

	t.Run("returns error if invalid response json", func (t *testing.T) {
		returnData = new(bytes.Buffer)
		_, err = fmt.Fprint(returnData, "not json")
		if err != nil {
			t.Logf("error writing to buffer: %v", err)
		}

		returnCode = http.StatusOK
		actualBuild, err := client.TriggerBuild("job", "master", nil)
		assert.EqualError(t, err, "error unmarshaling json: invalid character 'o' in literal null (expecting 'u')")
		assert.Nil(t, actualBuild)
	})
}
